import React, { useMemo } from 'react';
import { TrendingUp, TrendingDown, DollarSign, Home, Clock, Target } from 'lucide-react';
import { Property } from './types';

interface MarketIntelligenceProps {
  properties: Property[];
  selectedArea?: {
    bounds: { north: number; south: number; east: number; west: number };
    name: string;
  };
}

interface MarketMetrics {
  totalProperties: number;
  averagePrice: number;
  medianPrice: number;
  pricePerSqft: number;
  priceRange: { min: number; max: number };
  propertyTypes: Record<string, number>;
  listingTypes: Record<string, number>;
  marketTrend: 'up' | 'down' | 'stable';
  investmentScore: number;
  liquidityScore: number;
}

export function MarketIntelligence({ properties, selectedArea }: MarketIntelligenceProps) {
  const metrics = useMemo((): MarketMetrics => {
    if (properties.length === 0) {
      return {
        totalProperties: 0,
        averagePrice: 0,
        medianPrice: 0,
        pricePerSqft: 0,
        priceRange: { min: 0, max: 0 },
        propertyTypes: {},
        listingTypes: {},
        marketTrend: 'stable',
        investmentScore: 0,
        liquidityScore: 0
      };
    }

    // Filter properties in selected area if provided
    let filteredProperties = properties;
    if (selectedArea) {
      filteredProperties = properties.filter(p => 
        p.latitude >= selectedArea.bounds.south &&
        p.latitude <= selectedArea.bounds.north &&
        p.longitude >= selectedArea.bounds.west &&
        p.longitude <= selectedArea.bounds.east
      );
    }

    // Calculate price metrics
    const prices = filteredProperties
      .map(p => p.sale_price || (p.monthly_rent * 12) || 0)
      .filter(price => price > 0)
      .sort((a, b) => a - b);

    const averagePrice = prices.length > 0 ? prices.reduce((a, b) => a + b, 0) / prices.length : 0;
    const medianPrice = prices.length > 0 ? prices[Math.floor(prices.length / 2)] : 0;
    const priceRange = prices.length > 0 ? { min: prices[0], max: prices[prices.length - 1] } : { min: 0, max: 0 };

    // Calculate price per sqft
    const pricePerSqftValues = filteredProperties
      .filter(p => (p.built_up_area || p.carpet_area || p.super_area) && (p.sale_price || p.monthly_rent))
      .map(p => {
        const price = p.sale_price || (p.monthly_rent * 12);
        const area = p.built_up_area || p.carpet_area || p.super_area || 1;
        return price / area;
      });

    const avgPricePerSqft = pricePerSqftValues.length > 0 
      ? pricePerSqftValues.reduce((a, b) => a + b, 0) / pricePerSqftValues.length 
      : 0;

    // Property type distribution
    const propertyTypes = filteredProperties.reduce((acc, p) => {
      acc[p.property_type] = (acc[p.property_type] || 0) + 1;
      return acc;
    }, {} as Record<string, number>);

    // Listing type distribution
    const listingTypes = filteredProperties.reduce((acc, p) => {
      acc[p.listing_type] = (acc[p.listing_type] || 0) + 1;
      return acc;
    }, {} as Record<string, number>);

    // Calculate investment and liquidity scores (simplified)
    const investmentScore = Math.min(100, Math.max(0, 
      (filteredProperties.length / 10) * 20 + // Density factor
      (averagePrice > 5000000 ? 30 : averagePrice > 2000000 ? 20 : 10) + // Price factor
      (avgPricePerSqft > 5000 ? 25 : avgPricePerSqft > 3000 ? 15 : 5) // Price/sqft factor
    ));

    const liquidityScore = Math.min(100, Math.max(0,
      (listingTypes['sell'] || 0) / filteredProperties.length * 40 + // Sale listings
      (filteredProperties.length > 20 ? 30 : filteredProperties.length > 10 ? 20 : 10) + // Market activity
      (propertyTypes['apartment'] || 0) / filteredProperties.length * 30 // Apartment liquidity
    ));

    // Simple trend calculation (would need historical data for real implementation)
    const marketTrend: 'up' | 'down' | 'stable' = investmentScore > 70 ? 'up' : investmentScore < 40 ? 'down' : 'stable';

    return {
      totalProperties: filteredProperties.length,
      averagePrice,
      medianPrice,
      pricePerSqft: avgPricePerSqft,
      priceRange,
      propertyTypes,
      listingTypes,
      marketTrend,
      investmentScore,
      liquidityScore
    };
  }, [properties, selectedArea]);

  const formatPrice = (price: number): string => {
    if (price >= 10000000) return `₹${(price / 10000000).toFixed(1)}Cr`;
    if (price >= 100000) return `₹${(price / 100000).toFixed(1)}L`;
    return `₹${price.toLocaleString()}`;
  };

  const getScoreColor = (score: number): string => {
    if (score >= 70) return 'text-green-600 bg-green-50';
    if (score >= 40) return 'text-yellow-600 bg-yellow-50';
    return 'text-red-600 bg-red-50';
  };

  const getTrendIcon = () => {
    switch (metrics.marketTrend) {
      case 'up': return <TrendingUp className="w-4 h-4 text-green-600" />;
      case 'down': return <TrendingDown className="w-4 h-4 text-red-600" />;
      default: return <div className="w-4 h-4 bg-gray-400 rounded-full" />;
    }
  };

  if (metrics.totalProperties === 0) {
    return (
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Market Intelligence</h3>
        <div className="text-center text-gray-500 py-8">
          <Home className="w-12 h-12 mx-auto mb-3 text-gray-300" />
          <p>No properties available for analysis</p>
          <p className="text-sm">Select an area or adjust filters to view market insights</p>
        </div>
      </div>
    );
  }

  return (
    <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
      <div className="flex items-center justify-between mb-6">
        <h3 className="text-lg font-semibold text-gray-900">Market Intelligence</h3>
        {selectedArea && (
          <span className="text-sm text-gray-600 bg-gray-100 px-2 py-1 rounded">
            {selectedArea.name}
          </span>
        )}
      </div>

      {/* Key Metrics */}
      <div className="grid grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
        <div className="text-center p-3 bg-gray-50 rounded-lg">
          <div className="text-2xl font-bold text-gray-900">{metrics.totalProperties}</div>
          <div className="text-xs text-gray-600">Properties</div>
        </div>
        
        <div className="text-center p-3 bg-gray-50 rounded-lg">
          <div className="text-lg font-bold text-gray-900">{formatPrice(metrics.averagePrice)}</div>
          <div className="text-xs text-gray-600">Avg Price</div>
        </div>
        
        <div className="text-center p-3 bg-gray-50 rounded-lg">
          <div className="text-lg font-bold text-gray-900">₹{Math.round(metrics.pricePerSqft).toLocaleString()}</div>
          <div className="text-xs text-gray-600">Per Sq Ft</div>
        </div>
        
        <div className="text-center p-3 bg-gray-50 rounded-lg">
          <div className="flex items-center justify-center gap-1">
            {getTrendIcon()}
            <span className="text-lg font-bold text-gray-900">
              {metrics.marketTrend === 'up' ? '+' : metrics.marketTrend === 'down' ? '-' : ''}
            </span>
          </div>
          <div className="text-xs text-gray-600">Market Trend</div>
        </div>
      </div>

      {/* Price Range */}
      <div className="mb-6">
        <h4 className="text-sm font-medium text-gray-700 mb-2">Price Range</h4>
        <div className="flex items-center gap-4 text-sm">
          <span className="text-gray-600">Min: <span className="font-bold text-base text-gray-900">{formatPrice(metrics.priceRange.min)}</span></span>
          <div className="flex-1 h-2 bg-gray-200 rounded-full overflow-hidden">
            <div className="h-full bg-gradient-to-r from-green-400 to-red-400"></div>
          </div>
          <span className="text-gray-600">Max: <span className="font-bold text-base text-gray-900">{formatPrice(metrics.priceRange.max)}</span></span>
        </div>
      </div>

      {/* Investment Scores */}
      <div className="grid grid-cols-2 gap-4 mb-6">
        <div className="p-3 border border-gray-200 rounded-lg">
          <div className="flex items-center gap-2 mb-2">
            <Target className="w-4 h-4 text-gray-600" />
            <span className="text-sm font-medium text-gray-700">Investment Score</span>
          </div>
          <div className={`text-lg font-bold px-2 py-1 rounded ${getScoreColor(metrics.investmentScore)}`}>
            {Math.round(metrics.investmentScore)}/100
          </div>
        </div>
        
        <div className="p-3 border border-gray-200 rounded-lg">
          <div className="flex items-center gap-2 mb-2">
            <Clock className="w-4 h-4 text-gray-600" />
            <span className="text-sm font-medium text-gray-700">Liquidity Score</span>
          </div>
          <div className={`text-lg font-bold px-2 py-1 rounded ${getScoreColor(metrics.liquidityScore)}`}>
            {Math.round(metrics.liquidityScore)}/100
          </div>
        </div>
      </div>

      {/* Property Distribution */}
      <div className="space-y-4">
        <div>
          <h4 className="text-sm font-medium text-gray-700 mb-2">Property Types</h4>
          <div className="space-y-2">
            {Object.entries(metrics.propertyTypes).map(([type, count]) => (
              <div key={type} className="flex items-center justify-between text-sm">
                <span className="capitalize text-gray-600">{type}</span>
                <div className="flex items-center gap-2">
                  <div className="w-16 h-2 bg-gray-200 rounded-full overflow-hidden">
                    <div 
                      className="h-full bg-blue-500"
                      style={{ width: `${(count / metrics.totalProperties) * 100}%` }}
                    ></div>
                  </div>
                  <span className="text-gray-900 font-medium">{count}</span>
                </div>
              </div>
            ))}
          </div>
        </div>

        <div>
          <h4 className="text-sm font-medium text-gray-700 mb-2">Listing Types</h4>
          <div className="space-y-2">
            {Object.entries(metrics.listingTypes).map(([type, count]) => (
              <div key={type} className="flex items-center justify-between text-sm">
                <span className="capitalize text-gray-600">{type}</span>
                <div className="flex items-center gap-2">
                  <div className="w-16 h-2 bg-gray-200 rounded-full overflow-hidden">
                    <div 
                      className="h-full bg-green-500"
                      style={{ width: `${(count / metrics.totalProperties) * 100}%` }}
                    ></div>
                  </div>
                  <span className="text-gray-900 font-medium">{count}</span>
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>
    </div>
  );
}

export default MarketIntelligence;