import React, { useState, useEffect, useRef, useMemo } from 'react';
import { useNavigate } from 'react-router-dom';
import { HeatmapControls } from './HeatmapControls';
import { HeatmapLegend } from './HeatmapLegend';
import { HeatmapInterpretation } from './HeatmapInterpretation';
import { HeatmapErrorBoundary } from './HeatmapErrorBoundary';
import HeatmapFilters, { FilterState } from './HeatmapFilters';
import InteractiveControls, { AreaSelectionOverlay } from './InteractiveControls';
import MarketIntelligence from './MarketIntelligence';
import AreaComparison from './AreaComparison';
import { HeatmapMode, Property, HeatmapPoint } from './types';
import { X, MapPin, Home, AlertCircle } from 'lucide-react';
import { googleMapsService } from '../../utils/googleMaps';
import { heatmapDataService } from '../../services/heatmapDataService';
import HeatmapAlgorithms from '../../utils/heatmapAlgorithms';
import MarkerClustering from '../../utils/markerClustering';

// Google Maps type declarations
declare global {
  interface Window {
    google: any;
  }
}

interface PropertyHeatmapProps {
  properties: Property[];
}

export function PropertyHeatmap({ properties }: PropertyHeatmapProps) {
  const navigate = useNavigate();
  const [mode, setMode] = useState<HeatmapMode>('density');
  const [center, setCenter] = useState<[number, number]>([20.5937, 78.9629]);
  const [heatmapPoints, setHeatmapPoints] = useState<HeatmapPoint[]>([]);
  const [selectedProperty, setSelectedProperty] = useState<Property | null>(null);
  const [popupPosition, setPopupPosition] = useState<{ x: number; y: number } | null>(null);
  const [zoomLevel, setZoomLevel] = useState<number>(5);
  const [isLoading, setIsLoading] = useState<boolean>(false);
  const [filters, setFilters] = useState<FilterState | null>(null);
  const [areaSelectMode, setAreaSelectMode] = useState<'rectangle' | 'circle' | null>(null);
  const [selectedArea, setSelectedArea] = useState<any>(null);
  const [showComparison, setShowComparison] = useState<boolean>(false);
  const mapRef = useRef<HTMLDivElement>(null);
  const mapInstanceRef = useRef<any>(null);
  const markersRef = useRef<any[]>([]);

  const filteredProperties = useMemo(() => {
    if (!filters) return properties;
    
    return properties.filter(property => {
      if (filters.propertyType && property.property_type !== filters.propertyType) {
        return false;
      }
      
      if (filters.listingType && property.listing_type !== filters.listingType) {
        return false;
      }
      
      const price = property.sale_price || (property.monthly_rent * 12) || 0;
      if (filters.priceRange.min && price < filters.priceRange.min) return false;
      if (filters.priceRange.max && price > filters.priceRange.max) return false;
      
      return true;
    });
  }, [properties, filters]);

  useEffect(() => {
    // Clear cache to get fresh data
    heatmapDataService.clearCache();
    
    generateHeatmapPoints();
  }, [filteredProperties, mode]);

  useEffect(() => {
    const initMap = async () => {
      try {
        await googleMapsService.loadGoogleMaps();
        
        if (mapRef.current && googleMapsService.isGoogleMapsLoaded()) {
          mapInstanceRef.current = new window.google.maps.Map(mapRef.current, {
            center: { lat: center[0], lng: center[1] },
            zoom: zoomLevel,
            styles: [
              {
                featureType: 'all',
                elementType: 'labels',
                stylers: [{ visibility: 'on' }]
              }
            ],
            disableDefaultUI: false,
            zoomControl: true,
            mapTypeControl: false,
            scaleControl: false,
            streetViewControl: false,
            rotateControl: false,
            fullscreenControl: false
          });
          
          // Add zoom change listener
          mapInstanceRef.current.addListener('zoom_changed', () => {
            const newZoom = mapInstanceRef.current.getZoom();
            if (newZoom !== zoomLevel) {
              setZoomLevel(newZoom);
            }
          });
        }
      } catch (error) {
        console.error('Failed to initialize Google Maps:', error);
      }
    };

    initMap();
  }, [center]);

  useEffect(() => {
    if (mapInstanceRef.current && heatmapPoints.length > 0) {
      // Clear existing markers efficiently
      MarkerClustering.cleanupMarkers(markersRef.current);
      markersRef.current = [];

      // Get viewport bounds for clustering
      const viewport = MarkerClustering.getViewportBounds(mapInstanceRef.current);
      
      // Cluster markers for performance
      const clusters = MarkerClustering.clusterMarkers(heatmapPoints, zoomLevel, viewport);
      
      // Create simple markers for small datasets
      const newMarkers = heatmapPoints.map(point => {
        const marker = new window.google.maps.Marker({
          position: { lat: point.lat, lng: point.lng },
          map: mapInstanceRef.current,
          icon: {
            path: window.google.maps.SymbolPath.CIRCLE,
            scale: 8, // Size of the dot
            fillColor: '#FF0000', // Red color
            fillOpacity: 1.0,
            strokeColor: '#FF0000', // Red color
            strokeWeight: 2,
            strokeOpacity: 1.0
          },
          clickable: true
        });

        marker.addListener('click', (event: any) => {
          console.log('Marker clicked at:', point.lat, point.lng);
          console.log('Available properties:', filteredProperties.length);

          // Find property by exact coordinates or closest match
          let property = filteredProperties.find(p =>
            p.latitude === point.lat && p.longitude === point.lng
          );

          // If exact match not found, find closest property
          if (!property) {
            property = filteredProperties.find(p =>
              Math.abs(p.latitude - point.lat) < 0.01 &&
              Math.abs(p.longitude - point.lng) < 0.01
            );
          }

          // If still not found, use first property as fallback
          if (!property && filteredProperties.length > 0) {
            property = filteredProperties[0];
          }

          console.log('Found property:', property);

          if (property) {
            setSelectedProperty(property);
            setPopupPosition({ x: 300, y: 200 });
            console.log('Property popup should show now');
          } else {
            console.log('No property found for this point');
          }
        });

        return marker;
      });
      
      markersRef.current = newMarkers;
    }
  }, [heatmapPoints, mode, zoomLevel]);

  const generateHeatmapPoints = () => {
    const propsToUse = filteredProperties;
    
    if (propsToUse.length === 0) {
      setHeatmapPoints([]);
      return;
    }

    setIsLoading(true);
    
    try {
      let points: HeatmapPoint[] = [];
      
      // For small datasets, show individual properties
      if (propsToUse.length <= 10 || mode !== 'density') {
        points = propsToUse.map(property => {
          const price = property.sale_price || (property.monthly_rent * 12) || 0;
          return {
            lat: property.latitude,
            lng: property.longitude,
            intensity: mode === 'density' ? 0.8 : Math.min(price / 10000000, 1),
            price,
            propertyCount: 1
          };
        });
      } else {
        // Use algorithms for larger datasets
        if (mode === 'density') {
          points = HeatmapAlgorithms.calculateDensityHeatmap(propsToUse, zoomLevel);
        } else if (mode === 'price') {
          points = HeatmapAlgorithms.calculatePriceHeatmap(propsToUse);
        } else if (mode === 'price_per_sqft') {
          points = HeatmapAlgorithms.calculatePricePerSqftHeatmap(propsToUse);
        } else if (mode === 'market_velocity') {
          points = HeatmapAlgorithms.calculatePriceHeatmap(propsToUse);
        } else {
          points = HeatmapAlgorithms.calculateDensityHeatmap(propsToUse, zoomLevel);
        }
      }
      
      console.log('Generated heatmap points:', points.length, 'from properties:', propsToUse.length);
      console.log('Sample point:', points[0]);
      console.log('Sample property:', propsToUse[0]);
      setHeatmapPoints(points);
    } catch (error) {
      console.error('Error generating heatmap points:', error);
      // Fallback: show properties as individual points
      const fallbackPoints = propsToUse.map(property => ({
        lat: property.latitude,
        lng: property.longitude,
        intensity: 0.6,
        price: property.sale_price || (property.monthly_rent * 12) || 0,
        propertyCount: 1
      }));
      setHeatmapPoints(fallbackPoints);
    } finally {
      setIsLoading(false);
    }
  };

  // Cleanup on unmount
  useEffect(() => {
    return () => {
      if (markersRef.current.length > 0) {
        MarkerClustering.cleanupMarkers(markersRef.current);
      }
    };
  }, []);

  const handleZoomIn = () => {
    if (mapInstanceRef.current && zoomLevel < 18) {
      mapInstanceRef.current.setZoom(zoomLevel + 1);
    }
  };

  const handleZoomOut = () => {
    if (mapInstanceRef.current && zoomLevel > 3) {
      mapInstanceRef.current.setZoom(zoomLevel - 1);
    }
  };

  const handleResetView = () => {
    if (mapInstanceRef.current) {
      mapInstanceRef.current.setCenter({ lat: 20.5937, lng: 78.9629 });
      mapInstanceRef.current.setZoom(5);
    }
  };

  const handleAreaSelect = (mode: 'rectangle' | 'circle' | null) => {
    setAreaSelectMode(mode);
  };

  const handleBookmarkArea = () => {
    if (mapInstanceRef.current) {
      const bounds = mapInstanceRef.current.getBounds();
      const bookmark = {
        name: `Area ${Date.now()}`,
        bounds: {
          north: bounds.getNorthEast().lat(),
          south: bounds.getSouthWest().lat(),
          east: bounds.getNorthEast().lng(),
          west: bounds.getSouthWest().lng()
        },
        zoom: zoomLevel
      };
      
      const bookmarks = JSON.parse(localStorage.getItem('heatmap_bookmarks') || '[]');
      bookmarks.push(bookmark);
      localStorage.setItem('heatmap_bookmarks', JSON.stringify(bookmarks));
      
      alert('Area bookmarked successfully!');
    }
  };

  const handleExportData = () => {
    const exportData = {
      timestamp: new Date().toISOString(),
      mode,
      properties: properties.length,
      heatmapPoints: heatmapPoints.length,
      filters,
      center,
      zoomLevel
    };
    
    const blob = new Blob([JSON.stringify(exportData, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `heatmap-analysis-${Date.now()}.json`;
    a.click();
    URL.revokeObjectURL(url);
  };



  return (
    <HeatmapErrorBoundary>
      <div className="w-full space-y-6">
        <HeatmapControls mode={mode} onModeChange={setMode} />
        
        <HeatmapFilters 
          onFiltersChange={setFilters} 
          propertyCount={filteredProperties.length}
        />
        
        <div className="space-y-6">
      
            <div className="relative">
              <div
                ref={mapRef}
                style={{ height: '500px', width: '100%' }}
                className="rounded-lg border border-gray-200"
              />

              <InteractiveControls
                onZoomIn={handleZoomIn}
                onZoomOut={handleZoomOut}
                onResetView={handleResetView}
                onAreaSelect={handleAreaSelect}
                onBookmarkArea={handleBookmarkArea}
                onExportData={handleExportData}
                isAreaSelectMode={areaSelectMode}
                currentZoom={zoomLevel}
              />

              <div className="absolute top-4 right-4 z-[40]">
                <HeatmapLegend mode={mode} />
              </div>
              
              <AreaSelectionOverlay 
                mode={areaSelectMode}
                onCancel={() => setAreaSelectMode(null)}
              />

              {/* Property Popup - Centered in Map Container */}
              {selectedProperty && (
                <div 
                  className="absolute z-[9999] transform -translate-x-1/2 -translate-y-1/2"
                  style={{ 
                    left: '50%', 
                    top: '50%'
                  }}
                >
                  <div className="bg-white rounded-xl shadow-2xl border border-gray-200 p-4 w-80 max-w-[90vw] max-h-[70vh] overflow-y-auto animate-in slide-in-from-bottom-2 duration-300">
                    {/* Close button */}
                    <button
                      onClick={() => setSelectedProperty(null)}
                      className="absolute top-2 right-2 p-1 rounded-full hover:bg-gray-100 transition-colors"
                    >
                      <X className="w-4 h-4 text-gray-500" />
                    </button>

                    {/* Property Image */}
                    <div className="w-full h-32 bg-gray-100 rounded-lg mb-3 overflow-hidden flex-shrink-0">
                      {(selectedProperty as any).primary_image ? (
                        <img
                          src={(selectedProperty as any).primary_image}
                          alt={selectedProperty.property_type}
                          className="w-full h-full object-cover"
                          onError={(e) => {
                            e.currentTarget.style.display = 'none';
                            const nextSibling = e.currentTarget.nextElementSibling as HTMLElement;
                            if (nextSibling) {
                              nextSibling.style.display = 'flex';
                            }
                          }}
                        />
                      ) : null}
                      <div className="w-full h-full bg-gradient-to-br from-blue-50 to-indigo-100 flex items-center justify-center">
                        <Home className="w-10 h-10 text-blue-400" />
                      </div>
                    </div>

                    {/* Property Details */}
                    <div className="space-y-3">
                      <h3 className="font-semibold text-gray-900 text-base leading-tight">
                        {selectedProperty.property_type?.charAt(0).toUpperCase() + selectedProperty.property_type?.slice(1)} in {selectedProperty.city}
                      </h3>
                      
                      <div className="flex items-center text-sm text-gray-600">
                        <MapPin className="w-4 h-4 mr-1 flex-shrink-0" />
                        <span className="break-words">{selectedProperty.city}</span>
                      </div>

                      <div className="text-lg font-bold text-blue-600">
                        ₹{(selectedProperty.sale_price || selectedProperty.monthly_rent || 0).toLocaleString()}
                        {selectedProperty.monthly_rent && !selectedProperty.sale_price && '/month'}
                      </div>

                      <div className="grid grid-cols-2 gap-2 text-sm">
                        <div className="bg-gray-50 p-2 rounded">
                          <div className="text-gray-500 text-xs">Type</div>
                          <div className="font-medium capitalize">{selectedProperty.property_type}</div>
                        </div>
                        <div className="bg-gray-50 p-2 rounded">
                          <div className="text-gray-500 text-xs">For</div>
                          <div className="font-medium">
                            {selectedProperty.listing_type === 'sell' ? 'Sale' : 
                             selectedProperty.listing_type === 'rent' ? 'Rent' :
                             selectedProperty.listing_type === 'lease' ? 'Lease' :
                             selectedProperty.listing_type === 'urgent-sale' ? 'Urgent Sale' :
                             selectedProperty.listing_type === 'bidding' ? 'Bidding' : 'Sale'}
                          </div>
                        </div>
                      </div>
                      
                      {(selectedProperty as any).built_up_area && (
                        <div className="bg-blue-50 p-2 rounded">
                          <div className="text-blue-600 text-xs font-medium">Built-up Area</div>
                          <div className="text-blue-900 font-semibold">{(selectedProperty as any).built_up_area} sq ft</div>
                        </div>
                      )}
                    </div>

                    {/* View Details Button */}
                    <button
                      onClick={() => {
                        navigate(`/property/${selectedProperty.id}`);
                        setSelectedProperty(null);
                      }}
                      className="w-full mt-4 bg-blue-600 text-white py-2 px-4 rounded-lg text-sm font-medium hover:bg-blue-700 transition-colors"
                    >
                      View Details
                    </button>
                  </div>
                  
                  {/* Arrow pointer */}
                  <div className="absolute top-full left-1/2 transform -translate-x-1/2">
                    <div className="w-0 h-0 border-l-8 border-r-8 border-t-8 border-l-transparent border-r-transparent border-t-white"></div>
                  </div>
                </div>
              )}
            </div>

            <div className="flex justify-between items-center text-sm text-gray-600">
              <span>
                Showing {heatmapPoints.length} {mode === 'density' ? 'density zones' : 
                  mode === 'price_per_sqft' ? 'price/sqft points' : 
                  mode === 'market_velocity' ? 'velocity indicators' : 'price points'} 
                from {filteredProperties.length} properties
              </span>
              {isLoading && (
                <span className="flex items-center gap-2">
                  <div className="w-4 h-4 border-2 border-blue-600 border-t-transparent rounded-full animate-spin"></div>
                  Processing...
                </span>
              )}
            </div>
        </div>
        
        <MarketIntelligence 
          properties={filteredProperties}
          selectedArea={selectedArea}
        />
        
        <div className="space-y-4">
          <div className="flex gap-2">
            <button
              onClick={() => setShowComparison(!showComparison)}
              className={`px-4 py-2 text-sm rounded-lg transition-colors ${
                showComparison 
                  ? 'bg-blue-600 text-white' 
                  : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
              }`}
            >
              Area Comparison
            </button>
          </div>
          
          {showComparison && (
            <AreaComparison 
              properties={filteredProperties}
              onAreaSelect={(callback) => {
                const bounds = {
                  north: center[0] + 0.01,
                  south: center[0] - 0.01,
                  east: center[1] + 0.01,
                  west: center[1] - 0.01
                };
                callback(bounds);
              }}
            />
          )}
        </div>

        <HeatmapInterpretation />
      </div>
    </HeatmapErrorBoundary>
  );
}