import { useState } from 'react';
import { Plus, Calendar, Check } from 'lucide-react';
import { toast } from 'sonner';
import { useNavigate } from 'react-router-dom';

interface AddToItineraryButtonProps {
  propertyId: number;
  propertyTitle: string;
  variant?: 'button' | 'icon';
}

export const AddToItineraryButton = ({ propertyId, propertyTitle, variant = 'button' }: AddToItineraryButtonProps) => {
  const [showModal, setShowModal] = useState(false);
  const [selectedItinerary, setSelectedItinerary] = useState<string>('');
  const navigate = useNavigate();

  // Mock itineraries - replace with API call
  const existingItineraries = [
    { id: '1', title: 'Weekend Property Tour - Mumbai', date: '2024-02-15', propertyCount: 3 },
    { id: '2', title: 'Investment Properties - Pune', date: '2024-01-10', propertyCount: 2 }
  ];

  const handleAddToExisting = () => {
    if (!selectedItinerary) {
      toast.error('Please select an itinerary');
      return;
    }
    toast.success(`Added ${propertyTitle} to itinerary`);
    setShowModal(false);
  };

  const handleCreateNew = () => {
    localStorage.setItem('pendingItineraryProperty', JSON.stringify({ propertyId, propertyTitle }));
    navigate('/create-itinerary');
  };

  if (variant === 'icon') {
    return (
      <>
        <button onClick={() => setShowModal(true)} className="p-2 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors" title="Add to Itinerary">
          <Calendar className="w-5 h-5 text-gray-700" />
        </button>
        {showModal && <Modal />}
      </>
    );
  }

  const Modal = () => (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50" onClick={() => setShowModal(false)}>
      <div className="bg-white rounded-lg p-6 max-w-md w-full mx-4" onClick={(e) => e.stopPropagation()}>
        <div className="flex items-center justify-between mb-4">
          <h3 className="text-lg font-semibold text-gray-900">Add to Itinerary</h3>
          <button onClick={() => setShowModal(false)} className="text-gray-400 hover:text-gray-600">
            <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
            </svg>
          </button>
        </div>

        <p className="text-sm text-gray-600 mb-4">Adding: <span className="font-medium text-gray-900">{propertyTitle}</span></p>

        <div className="space-y-3 mb-4">
          <button onClick={handleCreateNew} className="w-full p-4 border-2 border-dashed border-blue-300 rounded-lg hover:border-blue-500 hover:bg-blue-50 transition-all flex items-center gap-3 text-left">
            <div className="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center">
              <Plus className="w-5 h-5 text-blue-600" />
            </div>
            <div>
              <p className="font-semibold text-gray-900">Create New Itinerary</p>
              <p className="text-sm text-gray-600">Start a new property tour</p>
            </div>
          </button>

          {existingItineraries.length > 0 && (
            <>
              <div className="relative">
                <div className="absolute inset-0 flex items-center">
                  <div className="w-full border-t border-gray-300"></div>
                </div>
                <div className="relative flex justify-center text-sm">
                  <span className="px-2 bg-white text-gray-500">Or add to existing</span>
                </div>
              </div>

              <div className="space-y-2 max-h-60 overflow-y-auto">
                {existingItineraries.map((itinerary) => (
                  <button key={itinerary.id} onClick={() => setSelectedItinerary(itinerary.id)} className={`w-full p-3 border-2 rounded-lg transition-all flex items-center gap-3 text-left ${selectedItinerary === itinerary.id ? 'border-blue-500 bg-blue-50' : 'border-gray-200 hover:border-gray-300'}`}>
                    <div className={`w-6 h-6 rounded-full border-2 flex items-center justify-center ${selectedItinerary === itinerary.id ? 'bg-blue-600 border-blue-600' : 'border-gray-300'}`}>
                      {selectedItinerary === itinerary.id && <Check className="w-4 h-4 text-white" />}
                    </div>
                    <div className="flex-1">
                      <p className="font-medium text-gray-900">{itinerary.title}</p>
                      <p className="text-sm text-gray-600">{new Date(itinerary.date).toLocaleDateString()} • {itinerary.propertyCount} properties</p>
                    </div>
                  </button>
                ))}
              </div>

              <button onClick={handleAddToExisting} disabled={!selectedItinerary} className="w-full bg-blue-600 text-white py-3 rounded-lg hover:bg-blue-700 transition-colors font-semibold disabled:opacity-50 disabled:cursor-not-allowed">
                Add to Selected Itinerary
              </button>
            </>
          )}
        </div>
      </div>
    </div>
  );

  return (
    <>
      <button onClick={() => setShowModal(true)} className="flex items-center gap-2 px-4 py-2 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors">
        <Calendar className="w-5 h-5" />
        <span>Add to Itinerary</span>
      </button>
      {showModal && <Modal />}
    </>
  );
};
