import { Calendar, MapPin, Clock, MoreVertical, Play, Edit, Copy, Trash2 } from 'lucide-react';
import { StatusBadge } from './StatusBadge';
import { Itinerary } from '../../types/itinerary';
import { useState } from 'react';

interface ItineraryCardProps {
  itinerary: Itinerary;
  onView: (id: string) => void;
  onEdit: (id: string) => void;
  onDuplicate: (id: string) => void;
  onDelete: (id: string) => void;
  onStartTour: (id: string) => void;
}

export const ItineraryCard = ({ itinerary, onView, onEdit, onDuplicate, onDelete, onStartTour }: ItineraryCardProps) => {
  const [showMenu, setShowMenu] = useState(false);

  return (
    <div className="bg-white rounded-lg border border-gray-200 p-4 hover:shadow-md transition-shadow cursor-pointer">
      <div onClick={() => onView(itinerary.id)}>
        <div className="flex items-start justify-between mb-3">
          <div className="flex-1">
            <h3 className="text-lg font-semibold text-gray-900">{itinerary.title}</h3>
            <div className="flex items-center gap-2 mt-1 text-sm text-gray-600">
              <Calendar className="w-4 h-4" />
              <span>{new Date(itinerary.date).toLocaleDateString('en-US', { weekday: 'short', month: 'short', day: 'numeric' })}</span>
            </div>
          </div>
          <div className="flex items-center gap-2">
            <StatusBadge status={itinerary.status} size="sm" />
            <div className="relative">
              <button onClick={(e) => { e.stopPropagation(); setShowMenu(!showMenu); }} className="p-1 hover:bg-gray-100 rounded">
                <MoreVertical className="w-5 h-5 text-gray-600" />
              </button>
              {showMenu && (
                <div className="absolute right-0 mt-1 w-40 bg-white rounded-lg shadow-lg border border-gray-200 py-1 z-10">
                  <button onClick={(e) => { e.stopPropagation(); onEdit(itinerary.id); setShowMenu(false); }} className="w-full px-4 py-2 text-left text-sm hover:bg-gray-50 flex items-center gap-2">
                    <Edit className="w-4 h-4" /> Edit
                  </button>
                  <button onClick={(e) => { e.stopPropagation(); onDuplicate(itinerary.id); setShowMenu(false); }} className="w-full px-4 py-2 text-left text-sm hover:bg-gray-50 flex items-center gap-2">
                    <Copy className="w-4 h-4" /> Duplicate
                  </button>
                  <button onClick={(e) => { e.stopPropagation(); onDelete(itinerary.id); setShowMenu(false); }} className="w-full px-4 py-2 text-left text-sm hover:bg-gray-50 flex items-center gap-2 text-red-600">
                    <Trash2 className="w-4 h-4" /> Delete
                  </button>
                </div>
              )}
            </div>
          </div>
        </div>

        <div className="flex items-center gap-4 text-sm text-gray-600 mb-3">
          <div className="flex items-center gap-1">
            <MapPin className="w-4 h-4" />
            <span>{itinerary.properties.length} properties</span>
          </div>
          <div className="flex items-center gap-1">
            <Clock className="w-4 h-4" />
            <span>{Math.floor(itinerary.totalDuration / 60)}h {itinerary.totalDuration % 60}m</span>
          </div>
        </div>

        <div className="flex gap-2 overflow-x-auto pb-2">
          {itinerary.properties.slice(0, 3).map((prop) => (
            <div key={prop.id} className="flex-shrink-0 w-20 h-20 rounded-lg overflow-hidden border border-gray-200">
              <img src={prop.photo} alt={prop.title} className="w-full h-full object-cover" />
            </div>
          ))}
          {itinerary.properties.length > 3 && (
            <div className="flex-shrink-0 w-20 h-20 rounded-lg bg-gray-100 flex items-center justify-center text-gray-600 text-sm font-medium">
              +{itinerary.properties.length - 3}
            </div>
          )}
        </div>
      </div>

      {itinerary.status === 'scheduled' && (
        <button onClick={(e) => { e.stopPropagation(); onStartTour(itinerary.id); }} className="w-full mt-3 bg-blue-600 text-white py-2 rounded-lg hover:bg-blue-700 transition-colors flex items-center justify-center gap-2">
          <Play className="w-4 h-4" />
          Start Tour
        </button>
      )}
    </div>
  );
};
