import { useState } from 'react';
import { Search, Plus, Check } from 'lucide-react';

interface Property {
  id: number;
  title: string;
  address: string;
  city: string;
  photo: string;
  price: string;
}

interface PropertySelectorProps {
  selectedProperties: number[];
  onToggle: (propertyId: number) => void;
}

export const PropertySelector = ({ selectedProperties, onToggle }: PropertySelectorProps) => {
  const [search, setSearch] = useState('');

  // Mock properties - replace with API call
  const properties: Property[] = [
    { id: 1, title: 'Luxury 3BHK Apartment', address: 'Bandra West', city: 'Mumbai', photo: '/public/3D.png', price: '₹2.5 Cr' },
    { id: 2, title: 'Modern Villa', address: 'Juhu', city: 'Mumbai', photo: '/public/3D.png', price: '₹5 Cr' },
    { id: 3, title: 'Penthouse Suite', address: 'Worli', city: 'Mumbai', photo: '/public/3D.png', price: '₹8 Cr' },
    { id: 4, title: '2BHK Apartment', address: 'Koregaon Park', city: 'Pune', photo: '/public/3D.png', price: '₹1.2 Cr' },
    { id: 5, title: 'Commercial Space', address: 'Hinjewadi', city: 'Pune', photo: '/public/3D.png', price: '₹3 Cr' }
  ];

  const filtered = properties.filter(p => 
    p.title.toLowerCase().includes(search.toLowerCase()) || 
    p.address.toLowerCase().includes(search.toLowerCase()) ||
    p.city.toLowerCase().includes(search.toLowerCase())
  );

  return (
    <div className="space-y-4">
      <div className="relative">
        <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
        <input type="text" value={search} onChange={(e) => setSearch(e.target.value)} placeholder="Search properties..." className="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" />
      </div>

      <div className="space-y-3 max-h-96 overflow-y-auto">
        {filtered.map((property) => {
          const isSelected = selectedProperties.includes(property.id);
          return (
            <div key={property.id} onClick={() => onToggle(property.id)} className={`flex items-center gap-4 p-4 border rounded-lg cursor-pointer transition-all ${isSelected ? 'border-blue-500 bg-blue-50' : 'border-gray-200 hover:border-gray-300'}`}>
              <img src={property.photo} alt={property.title} className="w-20 h-20 rounded-lg object-cover" />
              <div className="flex-1">
                <h4 className="font-semibold text-gray-900">{property.title}</h4>
                <p className="text-sm text-gray-600">{property.address}, {property.city}</p>
                <p className="text-sm font-semibold text-blue-600 mt-1">{property.price}</p>
              </div>
              <div className={`w-6 h-6 rounded-full border-2 flex items-center justify-center ${isSelected ? 'bg-blue-600 border-blue-600' : 'border-gray-300'}`}>
                {isSelected && <Check className="w-4 h-4 text-white" />}
              </div>
            </div>
          );
        })}
      </div>

      {filtered.length === 0 && (
        <div className="text-center py-8 text-gray-500">
          No properties found
        </div>
      )}
    </div>
  );
};
