import { useState } from 'react';
import { QrCode, MapPin, Camera, Check } from 'lucide-react';
import { toast } from 'sonner';

interface TourCheckInProps {
  propertyTitle: string;
  onCheckIn: (method: 'qr' | 'gps' | 'manual') => void;
}

export const TourCheckIn = ({ propertyTitle, onCheckIn }: TourCheckInProps) => {
  const [method, setMethod] = useState<'qr' | 'gps' | 'manual'>('gps');
  const [checking, setChecking] = useState(false);

  const handleCheckIn = async () => {
    setChecking(true);
    
    if (method === 'gps') {
      navigator.geolocation.getCurrentPosition(
        (position) => {
          toast.success('Location verified! Checking in...');
          setTimeout(() => {
            onCheckIn('gps');
            setChecking(false);
          }, 1000);
        },
        (error) => {
          toast.error('Unable to get location. Please try manual check-in.');
          setChecking(false);
        }
      );
    } else {
      setTimeout(() => {
        onCheckIn(method);
        setChecking(false);
      }, 1000);
    }
  };

  return (
    <div className="bg-white rounded-lg border border-gray-200 p-6">
      <h3 className="text-lg font-semibold text-gray-900 mb-4">Check-in to {propertyTitle}</h3>
      
      <div className="space-y-3 mb-6">
        <button onClick={() => setMethod('gps')} className={`w-full p-4 border-2 rounded-lg flex items-center gap-3 transition-all ${method === 'gps' ? 'border-blue-600 bg-blue-50' : 'border-gray-200 hover:border-gray-300'}`}>
          <MapPin className={`w-6 h-6 ${method === 'gps' ? 'text-blue-600' : 'text-gray-600'}`} />
          <div className="flex-1 text-left">
            <p className="font-semibold text-gray-900">GPS Verification</p>
            <p className="text-sm text-gray-600">Auto check-in when within 50m</p>
          </div>
          {method === 'gps' && <Check className="w-5 h-5 text-blue-600" />}
        </button>

        <button onClick={() => setMethod('qr')} className={`w-full p-4 border-2 rounded-lg flex items-center gap-3 transition-all ${method === 'qr' ? 'border-blue-600 bg-blue-50' : 'border-gray-200 hover:border-gray-300'}`}>
          <QrCode className={`w-6 h-6 ${method === 'qr' ? 'text-blue-600' : 'text-gray-600'}`} />
          <div className="flex-1 text-left">
            <p className="font-semibold text-gray-900">QR Code Scan</p>
            <p className="text-sm text-gray-600">Scan property QR code</p>
          </div>
          {method === 'qr' && <Check className="w-5 h-5 text-blue-600" />}
        </button>

        <button onClick={() => setMethod('manual')} className={`w-full p-4 border-2 rounded-lg flex items-center gap-3 transition-all ${method === 'manual' ? 'border-blue-600 bg-blue-50' : 'border-gray-200 hover:border-gray-300'}`}>
          <Camera className={`w-6 h-6 ${method === 'manual' ? 'text-blue-600' : 'text-gray-600'}`} />
          <div className="flex-1 text-left">
            <p className="font-semibold text-gray-900">Manual Check-in</p>
            <p className="text-sm text-gray-600">Upload photo proof</p>
          </div>
          {method === 'manual' && <Check className="w-5 h-5 text-blue-600" />}
        </button>
      </div>

      <button onClick={handleCheckIn} disabled={checking} className="w-full bg-blue-600 text-white py-3 rounded-lg hover:bg-blue-700 transition-colors font-semibold disabled:opacity-50">
        {checking ? 'Checking in...' : 'Check In Now'}
      </button>
    </div>
  );
};
