import { Calendar, Clock, User, MapPin, Phone, Mail, X, CheckCircle } from 'lucide-react';
import { VisitBooking } from '../../types/itinerary';

interface VisitConfirmationModalProps {
  isOpen: boolean;
  onClose: () => void;
  booking: Partial<VisitBooking>;
  property: any;
  onConfirm: () => void;
}

export const VisitConfirmationModal = ({ isOpen, onClose, booking, property, onConfirm }: VisitConfirmationModalProps) => {
  if (!isOpen) return null;

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      weekday: 'long',
      year: 'numeric',
      month: 'long',
      day: 'numeric'
    });
  };

  const formatTime = (timeString: string) => {
    return new Date(`2000-01-01T${timeString}`).toLocaleTimeString('en-US', {
      hour: 'numeric',
      minute: '2-digit',
      hour12: true
    });
  };

  return (
    <div className="fixed inset-0 bg-black/30 backdrop-blur-sm flex items-center justify-center z-50 p-4">
      <div className="bg-white rounded-2xl max-w-3xl w-full max-h-[85vh] overflow-y-auto shadow-2xl border border-gray-200 mx-4">
        <div className="p-6 border-b border-gray-200">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-3">
              <div className="w-10 h-10 bg-green-100 rounded-full flex items-center justify-center">
                <CheckCircle className="w-6 h-6 text-green-600" />
              </div>
              <h2 className="text-xl font-bold text-gray-900">Confirm Your Visit</h2>
            </div>
            <button onClick={onClose} className="p-2 hover:bg-gray-100 rounded-lg transition-colors">
              <X className="w-5 h-5 text-gray-500" />
            </button>
          </div>
        </div>

        <div className="p-6 space-y-6">
          {/* Property Info */}
          <div className="bg-gray-50 rounded-lg p-4">
            <h3 className="font-semibold text-gray-900 mb-2">{property?.title}</h3>
            <div className="flex items-center gap-2 text-gray-600">
              <MapPin className="w-4 h-4" />
              <span>{property?.address}, {property?.city}</span>
            </div>
          </div>

          {/* Visit Details */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div>
              <h4 className="font-medium text-gray-900 mb-3">Visit Schedule</h4>
              <div className="space-y-2">
                <div className="flex items-center gap-2 text-sm">
                  <Calendar className="w-4 h-4 text-blue-600" />
                  <span>{booking.visitDate ? formatDate(booking.visitDate) : 'Not specified'}</span>
                </div>
                <div className="flex items-center gap-2 text-sm">
                  <Clock className="w-4 h-4 text-blue-600" />
                  <span>{booking.arrivalTime ? formatTime(booking.arrivalTime) : 'Not specified'} ({booking.duration} minutes)</span>
                </div>
                <div className="flex items-center gap-2 text-sm">
                  <User className="w-4 h-4 text-blue-600" />
                  <span>Meeting with {booking.meetingWith}</span>
                </div>
              </div>
            </div>

            <div>
              <h4 className="font-medium text-gray-900 mb-3">Contact Information</h4>
              <div className="space-y-2">
                <div className="flex items-center gap-2 text-sm">
                  <User className="w-4 h-4 text-green-600" />
                  <span>{booking.contactName}</span>
                </div>
                <div className="flex items-center gap-2 text-sm">
                  <Phone className="w-4 h-4 text-green-600" />
                  <span>{booking.contactPhone}</span>
                </div>
                {booking.contactEmail && (
                  <div className="flex items-center gap-2 text-sm">
                    <Mail className="w-4 h-4 text-green-600" />
                    <span>{booking.contactEmail}</span>
                  </div>
                )}
              </div>
            </div>
          </div>

          {/* Visit Preferences */}
          <div>
            <h4 className="font-medium text-gray-900 mb-3">Visit Preferences</h4>
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
              <div>
                <span className="text-gray-600">Type:</span>
                <p className="font-medium capitalize">{booking.visitType}</p>
              </div>
              <div>
                <span className="text-gray-600">Group Size:</span>
                <p className="font-medium">{booking.groupSize}</p>
              </div>
              <div>
                <span className="text-gray-600">Transport:</span>
                <p className="font-medium capitalize">{booking.transportMode}</p>
              </div>
              <div>
                <span className="text-gray-600">Purpose:</span>
                <p className="font-medium capitalize">{booking.purposeOfVisit}</p>
              </div>
            </div>
          </div>

          {/* Areas to Visit */}
          {booking.areasToVisit && booking.areasToVisit.length > 0 && (
            <div>
              <h4 className="font-medium text-gray-900 mb-3">Areas to Visit</h4>
              <div className="flex flex-wrap gap-2">
                {booking.areasToVisit.map((area, index) => (
                  <span key={index} className="bg-blue-50 text-blue-700 px-3 py-1 rounded-full text-sm">
                    {area}
                  </span>
                ))}
              </div>
            </div>
          )}

          {/* Special Requests */}
          {booking.specialRequests && (
            <div>
              <h4 className="font-medium text-gray-900 mb-3">Special Requests</h4>
              <p className="text-sm text-gray-600 bg-gray-50 p-3 rounded-lg">{booking.specialRequests}</p>
            </div>
          )}

          {/* Important Notes */}
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
            <h4 className="font-medium text-blue-900 mb-2">Important Notes</h4>
            <ul className="text-sm text-blue-800 space-y-1">
              <li>• Please arrive 5-10 minutes before your scheduled time</li>
              <li>• Bring a valid ID for verification</li>
              <li>• You will receive a confirmation SMS/email shortly</li>
              {booking.covidPrecautions && <li>• COVID-19 safety protocols will be followed</li>}
              {booking.parkingRequired && <li>• Parking space has been arranged</li>}
            </ul>
          </div>
        </div>

        <div className="p-6 border-t border-gray-200">
          <div className="flex gap-4">
            <button
              onClick={onClose}
              className="flex-1 px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-medium"
            >
              Go Back
            </button>
            <button
              onClick={onConfirm}
              className="flex-1 px-6 py-3 bg-black text-white rounded-lg hover:bg-gray-800 transition-colors font-semibold"
            >
              Confirm Booking
            </button>
          </div>
        </div>
      </div>
    </div>
  );
};