import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '../ui/card';
import { Badge } from '../ui/badge';
import { Button } from '../ui/button';
import { Input } from '../ui/input';
import { Textarea } from '../ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '../ui/select';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from '../ui/dialog';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '../ui/tabs';
import { Progress } from '../ui/progress';
import { Play, Pause, Plus, Settings, Users, TrendingUp, Mail, MessageSquare } from 'lucide-react';

interface Campaign {
  id: number;
  name: string;
  description: string;
  campaign_type: string;
  status: string;
  enrollments_count: number;
  completion_rate: number;
  conversion_rate: number;
  created_at: string;
}

interface CampaignStep {
  id: number;
  campaign_id: number;
  step_order: number;
  name: string;
  communication_type: string;
  delay_days: number;
  delay_hours: number;
  is_active: boolean;
}

const LeadNurturingCampaigns: React.FC = () => {
  const [campaigns, setCampaigns] = useState<Campaign[]>([]);
  const [selectedCampaign, setSelectedCampaign] = useState<Campaign | null>(null);
  const [campaignSteps, setCampaignSteps] = useState<CampaignStep[]>([]);
  const [isCreateModalOpen, setIsCreateModalOpen] = useState(false);
  const [isStepModalOpen, setIsStepModalOpen] = useState(false);
  const [loading, setLoading] = useState(true);

  const [newCampaign, setNewCampaign] = useState({
    name: '',
    description: '',
    campaign_type: 'Drip',
    target_audience: {}
  });

  const [newStep, setNewStep] = useState({
    name: '',
    communication_type: 'Email',
    delay_days: 0,
    delay_hours: 0,
    template_id: ''
  });

  useEffect(() => {
    fetchCampaigns();
  }, []);

  const fetchCampaigns = async () => {
    // Mock data for campaigns
    setCampaigns([
      {
        id: 1,
        name: 'New Lead Welcome Series',
        description: 'Welcome series for new property inquiries',
        campaign_type: 'Drip',
        status: 'Active',
        enrollments_count: 245,
        completion_rate: 78.5,
        conversion_rate: 12.3,
        created_at: '2024-01-10T00:00:00Z'
      },
      {
        id: 2,
        name: 'Property Viewing Follow-up',
        description: 'Follow-up sequence after property viewing',
        campaign_type: 'Trigger',
        status: 'Active',
        enrollments_count: 89,
        completion_rate: 85.2,
        conversion_rate: 28.7,
        created_at: '2024-01-12T00:00:00Z'
      },
      {
        id: 3,
        name: 'Inactive Lead Re-engagement',
        description: 'Re-engage leads with no activity for 30 days',
        campaign_type: 'Trigger',
        status: 'Paused',
        enrollments_count: 156,
        completion_rate: 45.8,
        conversion_rate: 8.9,
        created_at: '2024-01-08T00:00:00Z'
      }
    ]);
    setLoading(false);
  };

  const fetchCampaignSteps = async (campaignId: number) => {
    // Mock data for campaign steps
    setCampaignSteps([
      {
        id: 1,
        campaign_id: campaignId,
        step_order: 1,
        name: 'Welcome Email',
        communication_type: 'Email',
        delay_days: 0,
        delay_hours: 1,
        is_active: true
      },
      {
        id: 2,
        campaign_id: campaignId,
        step_order: 2,
        name: 'Property Information SMS',
        communication_type: 'SMS',
        delay_days: 1,
        delay_hours: 0,
        is_active: true
      },
      {
        id: 3,
        campaign_id: campaignId,
        step_order: 3,
        name: 'Follow-up Call',
        communication_type: 'Call',
        delay_days: 2,
        delay_hours: 0,
        is_active: true
      }
    ]);
  };

  const createCampaign = async () => {
    // API call to create campaign
    setIsCreateModalOpen(false);
    setNewCampaign({
      name: '',
      description: '',
      campaign_type: 'Drip',
      target_audience: {}
    });
    fetchCampaigns();
  };

  const toggleCampaignStatus = async (campaignId: number, currentStatus: string) => {
    const newStatus = currentStatus === 'Active' ? 'Paused' : 'Active';
    // API call to update campaign status
    setCampaigns(campaigns.map(c => 
      c.id === campaignId ? { ...c, status: newStatus } : c
    ));
  };

  const addCampaignStep = async () => {
    if (!selectedCampaign) return;
    
    // API call to add campaign step
    setIsStepModalOpen(false);
    setNewStep({
      name: '',
      communication_type: 'Email',
      delay_days: 0,
      delay_hours: 0,
      template_id: ''
    });
    fetchCampaignSteps(selectedCampaign.id);
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'Active': return 'bg-green-500';
      case 'Paused': return 'bg-yellow-500';
      case 'Draft': return 'bg-gray-500';
      case 'Completed': return 'bg-blue-500';
      default: return 'bg-gray-500';
    }
  };

  const getTypeIcon = (type: string) => {
    switch (type) {
      case 'Email': return <Mail className="w-4 h-4" />;
      case 'SMS': return <MessageSquare className="w-4 h-4" />;
      case 'WhatsApp': return <MessageSquare className="w-4 h-4" />;
      default: return <Mail className="w-4 h-4" />;
    }
  };

  const formatDelay = (days: number, hours: number) => {
    if (days === 0 && hours === 0) return 'Immediate';
    if (days === 0) return `${hours}h`;
    if (hours === 0) return `${days}d`;
    return `${days}d ${hours}h`;
  };

  return (
    <div className="p-6 space-y-6">
      <div className="flex justify-between items-center">
        <h1 className="text-3xl font-bold">Lead Nurturing Campaigns</h1>
        <Dialog open={isCreateModalOpen} onOpenChange={setIsCreateModalOpen}>
          <DialogTrigger asChild>
            <Button>
              <Plus className="w-4 h-4 mr-2" />
              Create Campaign
            </Button>
          </DialogTrigger>
          <DialogContent className="max-w-2xl">
            <DialogHeader>
              <DialogTitle>Create New Campaign</DialogTitle>
            </DialogHeader>
            <div className="space-y-4">
              <Input
                placeholder="Campaign Name"
                value={newCampaign.name}
                onChange={(e) => setNewCampaign({...newCampaign, name: e.target.value})}
              />
              <Textarea
                placeholder="Description"
                value={newCampaign.description}
                onChange={(e) => setNewCampaign({...newCampaign, description: e.target.value})}
              />
              <Select value={newCampaign.campaign_type} onValueChange={(value) => setNewCampaign({...newCampaign, campaign_type: value})}>
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="Drip">Drip Campaign</SelectItem>
                  <SelectItem value="Trigger">Trigger Campaign</SelectItem>
                  <SelectItem value="Broadcast">Broadcast Campaign</SelectItem>
                </SelectContent>
              </Select>
              <Button onClick={createCampaign} className="w-full">
                Create Campaign
              </Button>
            </div>
          </DialogContent>
        </Dialog>
      </div>

      {/* Campaign Stats */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium">Active Campaigns</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">
              {campaigns.filter(c => c.status === 'Active').length}
            </div>
          </CardContent>
        </Card>
        
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium">Total Enrollments</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {campaigns.reduce((sum, c) => sum + c.enrollments_count, 0)}
            </div>
          </CardContent>
        </Card>
        
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium">Avg Completion Rate</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-blue-600">
              {campaigns.length > 0 ? (campaigns.reduce((sum, c) => sum + c.completion_rate, 0) / campaigns.length).toFixed(1) : 0}%
            </div>
          </CardContent>
        </Card>
        
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium">Avg Conversion Rate</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-purple-600">
              {campaigns.length > 0 ? (campaigns.reduce((sum, c) => sum + c.conversion_rate, 0) / campaigns.length).toFixed(1) : 0}%
            </div>
          </CardContent>
        </Card>
      </div>

      <Tabs defaultValue="campaigns" className="space-y-4">
        <TabsList>
          <TabsTrigger value="campaigns">Campaigns</TabsTrigger>
          <TabsTrigger value="performance">Performance</TabsTrigger>
          <TabsTrigger value="analytics">Analytics</TabsTrigger>
        </TabsList>

        <TabsContent value="campaigns" className="space-y-4">
          <div className="grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-3 gap-6">
            {campaigns.map((campaign) => (
              <Card key={campaign.id} className="cursor-pointer hover:shadow-lg transition-shadow">
                <CardHeader>
                  <div className="flex justify-between items-start">
                    <div>
                      <CardTitle className="text-lg">{campaign.name}</CardTitle>
                      <p className="text-sm text-gray-500 mt-1">{campaign.description}</p>
                    </div>
                    <Badge className={getStatusColor(campaign.status)}>
                      {campaign.status}
                    </Badge>
                  </div>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="flex justify-between text-sm">
                      <span>Type:</span>
                      <span className="font-medium">{campaign.campaign_type}</span>
                    </div>
                    
                    <div className="flex justify-between text-sm">
                      <span>Enrollments:</span>
                      <span className="font-medium flex items-center">
                        <Users className="w-4 h-4 mr-1" />
                        {campaign.enrollments_count}
                      </span>
                    </div>
                    
                    <div className="space-y-2">
                      <div className="flex justify-between text-sm">
                        <span>Completion Rate:</span>
                        <span className="font-medium">{campaign.completion_rate}%</span>
                      </div>
                      <Progress value={campaign.completion_rate} />
                    </div>
                    
                    <div className="space-y-2">
                      <div className="flex justify-between text-sm">
                        <span>Conversion Rate:</span>
                        <span className="font-medium text-green-600">{campaign.conversion_rate}%</span>
                      </div>
                      <Progress value={campaign.conversion_rate} className="h-2" />
                    </div>
                    
                    <div className="flex space-x-2 pt-2">
                      <Button
                        size="sm"
                        variant={campaign.status === 'Active' ? 'destructive' : 'default'}
                        onClick={() => toggleCampaignStatus(campaign.id, campaign.status)}
                      >
                        {campaign.status === 'Active' ? (
                          <>
                            <Pause className="w-4 h-4 mr-1" />
                            Pause
                          </>
                        ) : (
                          <>
                            <Play className="w-4 h-4 mr-1" />
                            Activate
                          </>
                        )}
                      </Button>
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => {
                          setSelectedCampaign(campaign);
                          fetchCampaignSteps(campaign.id);
                        }}
                      >
                        <Settings className="w-4 h-4 mr-1" />
                        Edit
                      </Button>
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </TabsContent>

        <TabsContent value="performance" className="space-y-4">
          {selectedCampaign ? (
            <div className="space-y-6">
              <div className="flex justify-between items-center">
                <h2 className="text-2xl font-bold">{selectedCampaign.name} - Campaign Steps</h2>
                <Dialog open={isStepModalOpen} onOpenChange={setIsStepModalOpen}>
                  <DialogTrigger asChild>
                    <Button>
                      <Plus className="w-4 h-4 mr-2" />
                      Add Step
                    </Button>
                  </DialogTrigger>
                  <DialogContent>
                    <DialogHeader>
                      <DialogTitle>Add Campaign Step</DialogTitle>
                    </DialogHeader>
                    <div className="space-y-4">
                      <Input
                        placeholder="Step Name"
                        value={newStep.name}
                        onChange={(e) => setNewStep({...newStep, name: e.target.value})}
                      />
                      <Select value={newStep.communication_type} onValueChange={(value) => setNewStep({...newStep, communication_type: value})}>
                        <SelectTrigger>
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="Email">Email</SelectItem>
                          <SelectItem value="SMS">SMS</SelectItem>
                          <SelectItem value="WhatsApp">WhatsApp</SelectItem>
                          <SelectItem value="Call">Call</SelectItem>
                        </SelectContent>
                      </Select>
                      <div className="grid grid-cols-2 gap-4">
                        <Input
                          type="number"
                          placeholder="Delay Days"
                          value={newStep.delay_days}
                          onChange={(e) => setNewStep({...newStep, delay_days: parseInt(e.target.value) || 0})}
                        />
                        <Input
                          type="number"
                          placeholder="Delay Hours"
                          value={newStep.delay_hours}
                          onChange={(e) => setNewStep({...newStep, delay_hours: parseInt(e.target.value) || 0})}
                        />
                      </div>
                      <Button onClick={addCampaignStep} className="w-full">
                        Add Step
                      </Button>
                    </div>
                  </DialogContent>
                </Dialog>
              </div>

              <div className="space-y-4">
                {campaignSteps.map((step, index) => (
                  <Card key={step.id}>
                    <CardContent className="p-4">
                      <div className="flex items-center justify-between">
                        <div className="flex items-center space-x-4">
                          <div className="w-8 h-8 bg-blue-500 text-white rounded-full flex items-center justify-center text-sm font-bold">
                            {step.step_order}
                          </div>
                          <div>
                            <div className="flex items-center space-x-2">
                              {getTypeIcon(step.communication_type)}
                              <h3 className="font-medium">{step.name}</h3>
                              <Badge variant="outline">{step.communication_type}</Badge>
                            </div>
                            <p className="text-sm text-gray-500 mt-1">
                              Delay: {formatDelay(step.delay_days, step.delay_hours)}
                            </p>
                          </div>
                        </div>
                        <div className="flex items-center space-x-2">
                          <Badge variant={step.is_active ? "default" : "secondary"}>
                            {step.is_active ? 'Active' : 'Inactive'}
                          </Badge>
                          <Button size="sm" variant="outline">
                            Edit
                          </Button>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </div>
          ) : (
            <Card>
              <CardContent className="p-8 text-center">
                <p className="text-gray-500">Select a campaign to view and edit its steps</p>
              </CardContent>
            </Card>
          )}
        </TabsContent>

        <TabsContent value="analytics" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>Campaign Analytics</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-center py-8 text-gray-500">
                <TrendingUp className="w-12 h-12 mx-auto mb-4 text-gray-400" />
                <p>Detailed analytics and performance metrics will be displayed here</p>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
};

export default LeadNurturingCampaigns;