import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '../ui/card';
import { Badge } from '../ui/badge';
import { Button } from '../ui/button';
import { Progress } from '../ui/progress';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '../ui/tabs';
import { BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, PieChart, Pie, Cell } from 'recharts';

interface LeadScore {
  id: number;
  lead_id: number;
  total_score: number;
  grade: string;
  score_breakdown: Record<string, any>;
  last_calculated: string;
}

interface ScoringRule {
  id: number;
  name: string;
  category: string;
  score_value: number;
  is_active: boolean;
}

const LeadScoringDashboard: React.FC = () => {
  const [leadScores, setLeadScores] = useState<LeadScore[]>([]);
  const [scoringRules, setScoringRules] = useState<ScoringRule[]>([]);
  const [scoreDistribution, setScoreDistribution] = useState<any>(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchLeadScores();
    fetchScoringRules();
    fetchScoreDistribution();
  }, []);

  const fetchLeadScores = async () => {
    // API call to fetch lead scores
    setLoading(false);
  };

  const fetchScoringRules = async () => {
    // API call to fetch scoring rules
  };

  const fetchScoreDistribution = async () => {
    // API call to fetch score distribution
    setScoreDistribution({
      grade_distribution: { A: 25, B: 45, C: 20, D: 10 },
      score_ranges: { '0-20': 10, '21-40': 20, '41-60': 30, '61-80': 25, '81-100': 15 },
      average_score: 58.5,
      total_leads: 150
    });
  };

  const recalculateAllScores = async () => {
    setLoading(true);
    // API call to recalculate all scores
    setTimeout(() => {
      setLoading(false);
      // Refresh data
      fetchLeadScores();
      fetchScoreDistribution();
    }, 2000);
  };

  const getGradeColor = (grade: string) => {
    switch (grade) {
      case 'A': return 'bg-green-500';
      case 'B': return 'bg-blue-500';
      case 'C': return 'bg-yellow-500';
      case 'D': return 'bg-red-500';
      default: return 'bg-gray-500';
    }
  };

  const gradeData = scoreDistribution ? Object.entries(scoreDistribution.grade_distribution).map(([grade, count]) => ({
    grade,
    count,
    percentage: ((count as number) / scoreDistribution.total_leads * 100).toFixed(1)
  })) : [];

  const scoreRangeData = scoreDistribution ? Object.entries(scoreDistribution.score_ranges).map(([range, count]) => ({
    range,
    count
  })) : [];

  const COLORS = ['#10B981', '#3B82F6', '#F59E0B', '#EF4444'];

  return (
    <div className="p-6 space-y-6">
      <div className="flex justify-between items-center">
        <h1 className="text-3xl font-bold">Lead Scoring Dashboard</h1>
        <Button onClick={recalculateAllScores} disabled={loading}>
          {loading ? 'Recalculating...' : 'Recalculate All Scores'}
        </Button>
      </div>

      {/* Overview Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium">Total Leads</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{scoreDistribution?.total_leads || 0}</div>
          </CardContent>
        </Card>
        
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium">Average Score</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{scoreDistribution?.average_score || 0}</div>
            <Progress value={scoreDistribution?.average_score || 0} className="mt-2" />
          </CardContent>
        </Card>
        
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium">High Quality (A+B)</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">
              {scoreDistribution ? scoreDistribution.grade_distribution.A + scoreDistribution.grade_distribution.B : 0}
            </div>
            <div className="text-sm text-gray-500">
              {scoreDistribution ? (((scoreDistribution.grade_distribution.A + scoreDistribution.grade_distribution.B) / scoreDistribution.total_leads * 100).toFixed(1)) : 0}%
            </div>
          </CardContent>
        </Card>
        
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium">Active Rules</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{scoringRules.filter(r => r.is_active).length}</div>
            <div className="text-sm text-gray-500">of {scoringRules.length} total</div>
          </CardContent>
        </Card>
      </div>

      <Tabs defaultValue="distribution" className="space-y-4">
        <TabsList>
          <TabsTrigger value="distribution">Score Distribution</TabsTrigger>
          <TabsTrigger value="rules">Scoring Rules</TabsTrigger>
          <TabsTrigger value="leads">Lead Scores</TabsTrigger>
        </TabsList>

        <TabsContent value="distribution" className="space-y-4">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {/* Grade Distribution Pie Chart */}
            <Card>
              <CardHeader>
                <CardTitle>Grade Distribution</CardTitle>
              </CardHeader>
              <CardContent>
                <ResponsiveContainer width="100%" height={300}>
                  <PieChart>
                    <Pie
                      data={gradeData}
                      cx="50%"
                      cy="50%"
                      labelLine={false}
                      label={({ grade, percentage }) => `${grade}: ${percentage}%`}
                      outerRadius={80}
                      fill="#8884d8"
                      dataKey="count"
                    >
                      {gradeData.map((entry, index) => (
                        <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                      ))}
                    </Pie>
                    <Tooltip />
                  </PieChart>
                </ResponsiveContainer>
              </CardContent>
            </Card>

            {/* Score Range Bar Chart */}
            <Card>
              <CardHeader>
                <CardTitle>Score Ranges</CardTitle>
              </CardHeader>
              <CardContent>
                <ResponsiveContainer width="100%" height={300}>
                  <BarChart data={scoreRangeData}>
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="range" />
                    <YAxis />
                    <Tooltip />
                    <Bar dataKey="count" fill="#3B82F6" />
                  </BarChart>
                </ResponsiveContainer>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="rules" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>Scoring Rules</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {scoringRules.map((rule) => (
                  <div key={rule.id} className="flex items-center justify-between p-4 border rounded-lg">
                    <div className="flex-1">
                      <h3 className="font-medium">{rule.name}</h3>
                      <p className="text-sm text-gray-500">{rule.category}</p>
                    </div>
                    <div className="flex items-center space-x-4">
                      <Badge variant="outline">+{rule.score_value} points</Badge>
                      <Badge variant={rule.is_active ? "default" : "secondary"}>
                        {rule.is_active ? 'Active' : 'Inactive'}
                      </Badge>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="leads" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>Recent Lead Scores</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {leadScores.slice(0, 10).map((score) => (
                  <div key={score.id} className="flex items-center justify-between p-4 border rounded-lg">
                    <div className="flex-1">
                      <h3 className="font-medium">Lead #{score.lead_id}</h3>
                      <p className="text-sm text-gray-500">
                        Last calculated: {new Date(score.last_calculated).toLocaleDateString()}
                      </p>
                    </div>
                    <div className="flex items-center space-x-4">
                      <div className="text-right">
                        <div className="text-lg font-bold">{score.total_score}</div>
                        <Progress value={score.total_score} className="w-20" />
                      </div>
                      <Badge className={getGradeColor(score.grade)}>
                        Grade {score.grade}
                      </Badge>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
};

export default LeadScoringDashboard;