// src/components/legal-assistance/AILegalAssistant.tsx

import React, { useState, useRef, useEffect } from 'react';
import { Button } from '../ui/button';
import { Input } from '../ui/input';
import { Card, CardContent, CardHeader, CardTitle } from '../ui/card';
import { Send, Bot, User, Loader2 } from 'lucide-react';
import { legalAiApi } from '../../api/legalAiApi';

interface ChatMessage {
  type: 'user' | 'ai';
  content: string;
  timestamp: Date;
}

const AILegalAssistant: React.FC = () => {
  const [messages, setMessages] = useState<ChatMessage[]>([]);
  const [inputValue, setInputValue] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const messagesEndRef = useRef<HTMLDivElement>(null);

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  };

  useEffect(() => {
    scrollToBottom();
  }, [messages]);

  const sendMessage = async () => {
    if (!inputValue.trim()) return;

    const userMessage: ChatMessage = {
      type: 'user',
      content: inputValue,
      timestamp: new Date()
    };

    setMessages(prev => [...prev, userMessage]);
    const currentInput = inputValue;
    setInputValue('');
    setIsLoading(true);

    try {
      const response = await legalAiApi.askLegalQuestion({
        question: currentInput,
        property_type: 'residential',
        location: 'India'
      });

      const aiMessage: ChatMessage = {
        type: 'ai',
        content: response.advice || 'Sorry, I could not process your question.',
        timestamp: new Date()
      };

      setMessages(prev => [...prev, aiMessage]);
    } catch (error) {
      const errorMessage: ChatMessage = {
        type: 'ai',
        content: 'Sorry, there was an error processing your question. Please try again.',
        timestamp: new Date()
      };
      setMessages(prev => [...prev, errorMessage]);
    } finally {
      setIsLoading(false);
    }
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      sendMessage();
    }
  };

  const quickQuestions = [
    "What documents do I need for property registration?",
    "How to check property title verification?",
    "What are the legal requirements for home loan?",
    "How to calculate EMI for my home loan?",
    "What is the loan eligibility criteria?",
    "How to verify property ownership?",
    "What is the process for property transfer?"
  ];

  return (
    <Card className="w-full max-w-4xl mx-auto h-[600px] flex flex-col">
      <CardHeader className="border-b">
        <CardTitle className="flex items-center gap-2">
          <Bot className="w-6 h-6 text-blue-600" />
          AI Legal &amp; Loan Assistant
        </CardTitle>
        <p className="text-sm text-gray-600">
          Get instant legal advice and loan assistance for property-related questions
        </p>
      </CardHeader>
      
      <CardContent className="flex-1 flex flex-col p-0">
        {/* Messages Area */}
        <div className="flex-1 overflow-y-auto p-4 space-y-4">
          {messages.length === 0 && (
            <div className="text-center py-8">
              <Bot className="w-16 h-16 text-gray-300 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-700 mb-2">
                Welcome to AI Legal &amp; Loan Assistant
              </h3>
              <p className="text-gray-500 mb-6">
                Ask me anything about property law, documentation, legal procedures, or loan assistance
              </p>
              
              {/* Quick Questions */}
              <div className="space-y-2">
                <p className="text-sm font-medium text-gray-600 mb-3">Quick Questions:</p>
                <div className="grid gap-2 max-w-md mx-auto">
                  {quickQuestions.map((question, index) => (
                    <button
                      key={index}
                      onClick={() => setInputValue(question)}
                      className="text-left p-2 text-sm bg-blue-50 hover:bg-blue-100 rounded-lg transition-colors"
                    >
                      {question}
                    </button>
                  ))}
                </div>
              </div>
            </div>
          )}
          
          {messages.map((message, index) => (
            <div
              key={index}
              className={flex gap-3 ${message.type === 'user' ? 'justify-end' : 'justify-start'}}
            >
              {message.type === 'ai' && (
                <div className="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center flex-shrink-0">
                  <Bot className="w-4 h-4 text-blue-600" />
                </div>
              )}
              
              <div
                className={`max-w-[70%] px-4 py-2 rounded-lg ${
                  message.type === 'user'
                    ? 'text-white'
                    : 'bg-gray-100 text-gray-800'
                }`}
                style={message.type === 'user' ? { backgroundColor: '#2563eb' } : {}}
              >
                <div className="text-sm whitespace-pre-wrap" dangerouslySetInnerHTML={{
                  __html: message.content
                    .replace(/\\(.?)\\*/g, '<strong>$1</strong>')
                    .replace(/\n/g, '<br/>')
                }} />
                <p className={`text-xs mt-1 ${
                  message.type === 'user' ? 'text-gray-300' : 'text-gray-500'
                }`}>
                  {message.timestamp.toLocaleTimeString()}
                </p>
              </div>
              
              {message.type === 'user' && (
                <div className="w-8 h-8 bg-gray-200 rounded-full flex items-center justify-center flex-shrink-0">
                  <User className="w-4 h-4 text-gray-600" />
                </div>
              )}
            </div>
          ))}
          
          {isLoading && (
            <div className="flex gap-3 justify-start">
              <div className="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center flex-shrink-0">
                <Bot className="w-4 h-4 text-blue-600" />
              </div>
              <div className="bg-gray-100 text-gray-800 max-w-[70%] px-4 py-2 rounded-lg">
                <div className="flex items-center gap-2">
                  <Loader2 className="w-4 h-4 animate-spin" />
                  <span className="text-sm">Thinking...</span>
                </div>
              </div>
            </div>
          )}
          
          <div ref={messagesEndRef} />
        </div>
        
        {/* Input Area */}
        <div className="border-t p-4">
          <div className="flex gap-2">
            <Input
              value={inputValue}
              onChange={(e) => setInputValue(e.target.value)}
              placeholder="Ask your legal or loan question..."
              onKeyPress={handleKeyPress}
              disabled={isLoading}
              className="flex-1"
            />
            <Button 
              onClick={sendMessage} 
              disabled={isLoading || !inputValue.trim()}
              size="icon"
              style={{ backgroundColor: '#2563eb', color: 'white' }}
            >
              <Send className="w-4 h-4" />
            </Button>
          </div>
          <p className="text-xs text-gray-500 mt-2">
            Press Enter to send • This is AI-generated advice, consult a lawyer for complex matters
          </p>
        </div>
      </CardContent>
    </Card>
  );
};

export default AILegalAssistant;