// src/components/legal-assistance/AILegalChat.tsx 

import React, { useState } from 'react';
import { Button } from '../ui/button';
import { Input } from '../ui/input';
import { Card, CardContent, CardHeader, CardTitle } from '../ui/card';
import { Send } from 'lucide-react';
import { legalAiApi } from '../../api/legalAiApi';

interface ChatMessage {
  type: 'user' | 'ai';
  content: string;
  timestamp: Date;
}

const AILegalChat: React.FC = () => {
  const [messages, setMessages] = useState<ChatMessage[]>([]);
  const [inputValue, setInputValue] = useState('');
  const [isLoading, setIsLoading] = useState(false);

  const sendMessage = async () => {
    if (!inputValue.trim()) return;

    const userMessage: ChatMessage = {
      type: 'user',
      content: inputValue,
      timestamp: new Date()
    };

    setMessages(prev => [...prev, userMessage]);
    const currentQuestion = inputValue;
    setInputValue('');
    setIsLoading(true);

    try {
      const response = await legalAiApi.askLegalQuestion({
        question: currentQuestion,
        property_type: 'residential',
        location: 'India'
      });

      const aiMessage: ChatMessage = {
        type: 'ai',
        content: response.advice || 'Sorry, I could not process your question.',
        timestamp: new Date()
      };

      setMessages(prev => [...prev, aiMessage]);
    } catch (error) {
      console.error('Legal AI Chat Error:', error);
      const errorMessage: ChatMessage = {
        type: 'ai',
        content: 'Sorry, I am unable to provide legal advice at the moment. Please try again later or contact our legal experts directly.',
        timestamp: new Date()
      };
      setMessages(prev => [...prev, errorMessage]);
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <Card className="w-full max-w-2xl mx-auto">
      <CardHeader>
        <CardTitle>AI Legal Assistant</CardTitle>
        <p className="text-sm text-gray-600">Ask questions about property law and legal procedures</p>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          <div className="h-96 overflow-y-auto border rounded-lg p-4 space-y-3">
            {messages.length === 0 && (
              <div className="text-center text-gray-500 mt-8">
                <p>Ask me anything about property law!</p>
                <p className="text-sm mt-2">Example: "What documents do I need for property registration?"</p>
              </div>
            )}
            
            {messages.map((message, index) => (
              <div
                key={index}
                className={flex ${message.type === 'user' ? 'justify-end' : 'justify-start'}}
              >
                <div
                  className={`max-w-xs lg:max-w-md px-4 py-2 rounded-lg ${
                    message.type === 'user'
                      ? 'text-white'
                      : 'bg-gray-100 text-gray-800'
                  }`}
                  style={message.type === 'user' ? { backgroundColor: '#2563eb !important' } : {}}
                >
                  <p className="text-sm">{message.content}</p>
                  <p className="text-xs mt-1 opacity-70">
                    {message.timestamp.toLocaleTimeString()}
                  </p>
                </div>
              </div>
            ))}
            
            {isLoading && (
              <div className="flex justify-start">
                <div className="bg-gray-100 text-gray-800 max-w-xs lg:max-w-md px-4 py-2 rounded-lg">
                  <p className="text-sm">Thinking...</p>
                </div>
              </div>
            )}
          </div>
          
          <div className="space-y-2">
            <div className="flex space-x-2">
              <Input
                value={inputValue}
                onChange={(e) => setInputValue(e.target.value)}
                placeholder="Ask your legal question..."
                onKeyPress={(e) => e.key === 'Enter' && sendMessage()}
                disabled={isLoading}
                className="flex-1"
              />
              <Button 
                onClick={sendMessage} 
                disabled={isLoading || !inputValue.trim()}
                className="bg-[#2563eb] hover:bg-[#1d4ed8] text-white px-3"
              >
                <Send className="w-4 h-4" />
              </Button>
            </div>
            <p className="text-xs text-gray-500 text-center">
              Press Enter to send • This is AI-generated advice, consult a lawyer for complex matters
            </p>
          </div>
        </div>
      </CardContent>
    </Card>
  );
};

export default AILegalChat;