import React, { useState, useEffect } from 'react';
import { Upload, FileText, CheckCircle, AlertTriangle, X, Trash2 } from 'lucide-react';

interface LegalDocumentUploadProps {
  propertyId: string | number;
  onUploadComplete?: () => void;
}

const LEGAL_DOCUMENT_TYPES = [
  { key: 'titleDeed', name: 'Title Deed / Mother Deed', required: true },
  { key: 'saleDeed', name: 'Sale Deed / Agreement to Sell', required: true },
  { key: 'encumbrance', name: 'Encumbrance Certificate (EC)', required: true },
  { key: 'taxReceipts', name: 'Property Tax Receipts', required: true },
  { key: 'khataPatta', name: 'Khata / Patta / Mutation Documents', required: true },
  { key: 'noc', name: 'NOCs (Society / Authority / Bank)', required: false },
  { key: 'occupancy', name: 'Completion / Occupancy Certificate', required: false },
  { key: 'builderDocs', name: 'Builder Documents (RERA, Allotment)', required: false },
  { key: 'idProof', name: 'Owner ID Proof (Aadhar Card)', required: true },
  { key: 'ownerAddressProof', name: 'Owner Address Proof', required: false },
];

export const LegalDocumentUpload: React.FC<LegalDocumentUploadProps> = ({
  propertyId,
  onUploadComplete
}) => {
  const [uploadedDocs, setUploadedDocs] = useState<Record<string, boolean>>({});
  const [documentIds, setDocumentIds] = useState<Record<string, number>>({});
  const [uploading, setUploading] = useState<string | null>(null);
  const [deleting, setDeleting] = useState<string | null>(null);
  const [error, setError] = useState<string | null>(null);
  const [loading, setLoading] = useState(true);

  // Check existing documents on component mount
  useEffect(() => {
    const checkExistingDocuments = async () => {
      if (!propertyId) {
        console.error('No property ID provided');
        setError('No property ID provided');
        setLoading(false);
        return;
      }

      try {
        const token = localStorage.getItem('sellerToken') || 
                     localStorage.getItem('token') || 
                     localStorage.getItem('accessToken');
        
        if (!token) {
          throw new Error('No authentication token found');
        }

        const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
        const response = await fetch(`${API_BASE}/properties/${propertyId}/documents`, {
          headers: {}
        });

        if (response.ok) {
          const data = await response.json();
          const existingDocs: Record<string, boolean> = {};
          const docIds: Record<string, number> = {};
          
          if (data.documents) {
            data.documents.forEach((doc: any) => {
              existingDocs[doc.doc_type] = true;
              docIds[doc.doc_type] = doc.id;
            });
          }
          
          setUploadedDocs(existingDocs);
          setDocumentIds(docIds);
        } else if (response.status === 403) {
          throw new Error('Access denied - you can only upload documents for your own properties');
        } else if (response.status === 404) {
          throw new Error('Property not found');
        } else {
          const errorData = await response.json().catch(() => ({}));
          throw new Error(errorData.detail || `HTTP ${response.status}`);
        }
      } catch (err) {
        console.error('Error checking existing documents:', err);
        setError(err instanceof Error ? err.message : 'Failed to load documents');
      } finally {
        setLoading(false);
      }
    };

    checkExistingDocuments();
  }, [propertyId]);

  const handleFileUpload = async (docType: string, file: File) => {
    setUploading(docType);
    setError(null);

    try {
      const token = localStorage.getItem('sellerToken') || 
                   localStorage.getItem('token') || 
                   localStorage.getItem('accessToken') ||
                   localStorage.getItem('authToken');
      
      console.log('Upload token:', token ? token.substring(0, 20) + '...' : 'None');
      
      if (!token) {
        throw new Error('No authentication token found');
      }

      const formData = new FormData();
      formData.append('file', file);
      formData.append('document_type', docType);

      const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
      const response = await fetch(`${API_BASE}/properties/${propertyId}/documents`, {
        method: 'POST',
        body: formData
      });

      if (response.ok) {
        const result = await response.json();
        setUploadedDocs(prev => ({ ...prev, [docType]: true }));
        setDocumentIds(prev => ({ ...prev, [docType]: result.document_id }));
        onUploadComplete?.();
      } else if (response.status === 403 || response.status === 401) {
        throw new Error('You can only upload documents for properties you own');
      } else {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.detail || `Upload failed with status ${response.status}`);
      }
    } catch (err) {
      setError(`Failed to upload ${docType}: ${err instanceof Error ? err.message : 'Unknown error'}`);
    } finally {
      setUploading(null);
    }
  };

  const handleDeleteDocument = async (docType: string) => {
    if (!documentIds[docType]) return;
    if (!confirm('Are you sure you want to delete this document?')) return;

    setDeleting(docType);
    setError(null);

    try {
      const token = localStorage.getItem('sellerToken') || 
                   localStorage.getItem('token') || 
                   localStorage.getItem('accessToken');
      
      if (!token) {
        throw new Error('No authentication token found');
      }

      const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
      const response = await fetch(`${API_BASE}/property-documents/${documentIds[docType]}`, {
        method: 'DELETE'
      });

      if (response.ok) {
        setUploadedDocs(prev => ({ ...prev, [docType]: false }));
        setDocumentIds(prev => {
          const newIds = { ...prev };
          delete newIds[docType];
          return newIds;
        });
        onUploadComplete?.();
      } else {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.detail || `Delete failed with status ${response.status}`);
      }
    } catch (err) {
      setError(`Failed to delete ${docType}: ${err instanceof Error ? err.message : 'Unknown error'}`);
    } finally {
      setDeleting(null);
    }
  };

  const getCompletionPercentage = () => {
    const uploaded = Object.values(uploadedDocs).filter(Boolean).length;
    return Math.round((uploaded / LEGAL_DOCUMENT_TYPES.length) * 100);
  };

  if (loading) {
    return (
      <div className="bg-white border border-gray-200 rounded-lg p-6">
        <div className="animate-pulse">
          <div className="h-6 bg-gray-200 rounded w-1/3 mb-4"></div>
          <div className="h-2 bg-gray-200 rounded mb-6"></div>
          <div className="space-y-4">
            {[1, 2, 3, 4, 5].map((i) => (
              <div key={i} className="h-16 bg-gray-200 rounded"></div>
            ))}
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="bg-white border border-gray-200 rounded-lg p-6">
      {/* Debug Info */}
      <div className="bg-blue-50 border border-blue-200 rounded-lg p-3 mb-4">
        <p className="text-sm text-blue-700">Property ID: {propertyId}</p>
        <p className="text-sm text-blue-700">Loading: {loading ? 'Yes' : 'No'}</p>
        <p className="text-sm text-blue-700">Documents found: {Object.keys(uploadedDocs).length}</p>
        <p className="text-sm text-blue-700">Error: {error || 'None'}</p>
      </div>
      
      <div className="mb-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-2">
          Legal Health Documents
        </h3>
        <div className="flex items-center gap-4">
          <div className="flex-1 bg-gray-200 rounded-full h-2">
            <div
              className="bg-blue-600 h-2 rounded-full transition-all duration-300"
              style={{ width: `${getCompletionPercentage()}%` }}
            />
          </div>
          <span className="text-sm font-medium text-gray-600">
            {getCompletionPercentage()}% Complete
          </span>
        </div>
      </div>

      {error && (
        <div className="mb-4 p-3 bg-red-50 border border-red-200 rounded-lg">
          <p className="text-sm text-red-800">{error}</p>
        </div>
      )}

      <div className="space-y-4">
        {LEGAL_DOCUMENT_TYPES.map((docType) => (
          <div
            key={docType.key}
            className="flex items-center justify-between p-4 border border-gray-200 rounded-lg"
          >
            <div className="flex items-center gap-3">
              {uploadedDocs[docType.key] ? (
                <CheckCircle className="w-5 h-5 text-green-600" />
              ) : (
                <FileText className="w-5 h-5 text-gray-400" />
              )}
              <div>
                <div className="font-medium text-gray-900">
                  {docType.name}
                  {docType.required && <span className="text-red-500 ml-1">*</span>}
                </div>
                <div className="text-sm text-gray-500">
                  {uploadedDocs[docType.key] ? 'Uploaded' : 'Not uploaded'}
                </div>
              </div>
            </div>

            <div className="flex items-center gap-2">
              {uploading === docType.key ? (
                <div className="flex items-center gap-2 text-blue-600">
                  <div className="w-4 h-4 border-2 border-blue-600 border-t-transparent rounded-full animate-spin" />
                  <span className="text-sm">Uploading...</span>
                </div>
              ) : deleting === docType.key ? (
                <div className="flex items-center gap-2 text-red-600">
                  <div className="w-4 h-4 border-2 border-red-600 border-t-transparent rounded-full animate-spin" />
                  <span className="text-sm">Deleting...</span>
                </div>
              ) : (
                <>
                  {uploadedDocs[docType.key] && (
                    <button
                      onClick={() => handleDeleteDocument(docType.key)}
                      className="px-2 py-1.5 bg-gray-100 hover:bg-gray-200 text-gray-700 rounded-lg transition-all duration-200 flex items-center"
                      title="Delete document"
                    >
                      <Trash2 className="w-3 h-3" />
                    </button>
                  )}
                  <label className="cursor-pointer">
                    <input
                      type="file"
                      accept=".pdf,.jpg,.jpeg,.png,.doc,.docx"
                      className="hidden"
                      onChange={(e) => {
                        const file = e.target.files?.[0];
                        if (file) {
                          handleFileUpload(docType.key, file);
                        }
                      }}
                    />
                    <div className="px-2 py-1.5 bg-blue-600 hover:bg-blue-700 text-white rounded-lg cursor-pointer border-2 border-blue-600 shadow-lg transition-all duration-200 hover:shadow-xl flex items-center" style={{backgroundColor: '#2563eb', color: 'white'}}>
                      <Upload className="w-3 h-3" />
                    </div>
                  </label>
                </>
              )}
            </div>
          </div>
        ))}
      </div>

      <div className="mt-6 p-4 bg-blue-50 border border-blue-200 rounded-lg">
        <p className="text-sm text-blue-800">
          <strong>Note:</strong> Upload all required documents (*) to improve your legal health score.
          Accepted formats: PDF, JPG, PNG, DOC, DOCX (Max 10MB each)
        </p>
      </div>
    </div>
  );
};