import React from 'react';
import { Shield, AlertTriangle, CheckCircle, FileText, Upload } from 'lucide-react';
import { LegalHealthData } from '../../hooks/useLegalHealth';

interface LegalHealthStatusProps {
  legalHealth: LegalHealthData;
  showDetails?: boolean;
  onUploadClick?: () => void;
  userRole?: 'seller' | 'agent' | 'buyer' | 'admin';
}

export const LegalHealthStatus: React.FC<LegalHealthStatusProps> = ({
  legalHealth,
  showDetails = false,
  onUploadClick,
  userRole = 'seller'
}) => {
  const getStatusIcon = () => {
    switch (legalHealth.legal_status) {
      case 'COMPLETE':
        return <CheckCircle className="w-5 h-5 text-green-600" />;
      case 'INCOMPLETE':
        return <AlertTriangle className="w-5 h-5 text-yellow-600" />;
      default:
        return <Shield className="w-5 h-5 text-gray-400" />;
    }
  };

  const getStatusColor = () => {
    switch (legalHealth.legal_status) {
      case 'COMPLETE':
        return 'bg-green-50 text-green-800 border-green-200';
      case 'INCOMPLETE':
        return 'bg-yellow-50 text-yellow-800 border-yellow-200';
      default:
        return 'bg-gray-50 text-gray-600 border-gray-200';
    }
  };

  const getRiskColor = () => {
    switch (legalHealth.risk_level) {
      case 'LOW':
        return 'text-green-600';
      case 'MEDIUM':
        return 'text-yellow-600';
      default:
        return 'text-red-600';
    }
  };

  const getRiskBadge = () => {
    const badges = {
      'HIGH': '🔴 High Risk',
      'MEDIUM': '🟡 Medium Risk', 
      'LOW': '🟢 Low Risk'
    };
    return badges[legalHealth.risk_level] || '⚪ Unknown Risk';
  };

  const getRiskExplanation = () => {
    const propertyType = legalHealth.property_type || 'property';
    const propertyDesc = `this ${propertyType.toLowerCase()}`;
    
    const explanations = {
      'HIGH': `Mandatory documents for ${propertyDesc} are missing.`,
      'MEDIUM': 'All mandatory documents uploaded. Some supporting documents are pending.',
      'LOW': `All required and supporting documents are uploaded for ${propertyDesc}.`
    };
    return explanations[legalHealth.risk_level] || 'Risk assessment unavailable.';
  };

  const getStatusText = () => {
    if (userRole === 'buyer') {
      return legalHealth.legal_status === 'COMPLETE' 
        ? 'Documents Provided' 
        : 'Documents Pending';
    }
    return legalHealth.legal_status.replace('_', ' ');
  };

  if (!showDetails) {
    // Compact view for property cards
    return (
      <div className={`inline-flex items-center gap-1 px-2 py-1 rounded-full text-xs border ${getStatusColor()}`}>
        {getStatusIcon()}
        <span>Legal: {getStatusText()}</span>
      </div>
    );
  }

  return (
    <div className="bg-white border border-gray-200 rounded-lg p-6">
      <div className="flex items-center justify-between mb-4">
        <h3 className="text-lg font-semibold text-gray-900 flex items-center gap-2">
          <Shield className="w-5 h-5 text-blue-600" />
          Legal Health & Title Status
        </h3>
        {userRole === 'seller' && onUploadClick && (
          <button
            onClick={onUploadClick}
            className="flex items-center gap-2 px-3 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 text-sm"
          >
            <Upload className="w-4 h-4" />
            Upload Documents
          </button>
        )}
      </div>

      {/* Legal Disclaimer */}
      <div className="bg-blue-50 border border-blue-200 rounded-lg p-3 mb-4">
        <p className="text-xs text-blue-800">
          <strong>Disclaimer:</strong> Legal information is provided by the seller for guidance only 
          and does not constitute legal verification or advice.
        </p>
      </div>

      {/* Status Overview */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
        <div className="text-center p-4 bg-gray-50 rounded-lg">
          <div className="flex items-center justify-center mb-2">
            {getStatusIcon()}
          </div>
          <div className="text-sm font-medium text-gray-900">Status</div>
          <div className="text-xs text-gray-600">{getStatusText()}</div>
        </div>
        
        <div className="text-center p-4 bg-gray-50 rounded-lg">
          <div className="text-2xl font-bold text-gray-900 mb-1">
            {legalHealth.uploaded_documents}/{legalHealth.required_documents}
          </div>
          <div className="text-sm font-medium text-gray-900">Documents</div>
          <div className="text-xs text-gray-600">
            {legalHealth.completion_percentage.toFixed(0)}% Complete
          </div>
        </div>

        {userRole !== 'buyer' && (
          <div className="text-center p-4 bg-gray-50 rounded-lg">
            <div className={`text-sm font-medium ${getRiskColor()}`}>
              {legalHealth.risk_level} RISK
            </div>
            <div className="text-xs text-gray-600 mt-1">Assessment</div>
          </div>
        )}
      </div>

      {/* Progress Bar */}
      <div className="mb-6">
        <div className="flex justify-between text-sm text-gray-600 mb-2">
          <span>Document Completion</span>
          <span>{legalHealth.completion_percentage.toFixed(0)}%</span>
        </div>
        <div className="w-full bg-gray-200 rounded-full h-2">
          <div
            className={`h-2 rounded-full ${
              legalHealth.completion_percentage === 100 
                ? 'bg-green-500' 
                : legalHealth.completion_percentage > 50 
                ? 'bg-yellow-500' 
                : 'bg-red-500'
            }`}
            style={{ width: `${legalHealth.completion_percentage}%` }}
          />
        </div>
      </div>

      {/* Document Checklist */}
      {userRole !== 'buyer' && (
        <div>
          <h4 className="text-sm font-medium text-gray-900 mb-3">Document Checklist</h4>
          <div className="space-y-2">
            {legalHealth.document_checklist.map((doc) => (
              <div key={doc.type} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                <div className="flex items-center gap-3">
                  {doc.uploaded ? (
                    <CheckCircle className="w-4 h-4 text-green-600" />
                  ) : (
                    <FileText className="w-4 h-4 text-gray-400" />
                  )}
                  <div>
                    <div className="text-sm font-medium text-gray-900">{doc.display_name}</div>
                    {doc.description && (
                      <div className="text-xs text-gray-500">{doc.description}</div>
                    )}
                  </div>
                </div>
                <div className={`text-xs px-2 py-1 rounded-full ${
                  doc.uploaded 
                    ? 'bg-green-100 text-green-800' 
                    : 'bg-red-100 text-red-800'
                }`}>
                  {doc.uploaded ? 'Uploaded' : 'Missing'}
                </div>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Risk Assessment for Sellers */}
      {userRole === 'seller' && (
        <div className="mt-6 p-4 bg-gray-50 rounded-lg">
          <div className="flex items-center justify-between mb-3">
            <h4 className="text-sm font-medium text-gray-900">Legal Risk Assessment</h4>
            <span className={`text-sm font-semibold ${getRiskColor()}`}>
              {getRiskBadge()}
            </span>
          </div>
          <p className="text-sm text-gray-600">
            {getRiskExplanation()}
          </p>
        </div>
      )}

      {/* Buyer View - Simplified */}
      {userRole === 'buyer' && (
        <div className="text-center p-4 bg-blue-50 rounded-lg">
          <p className="text-sm text-blue-800">
            {legalHealth.uploaded_documents} of {legalHealth.required_documents} required documents uploaded by seller
          </p>
        </div>
      )}
    </div>
  );
};