import React, { useState, useEffect } from 'react';
import { FileText } from 'lucide-react';

interface PropertyDocument {
  id: number;
  doc_type: string;
  display_name: string;
}

interface PropertyLegalDocumentsProps {
  propertyId: string | number;
}

export const PropertyLegalDocuments: React.FC<PropertyLegalDocumentsProps> = ({ propertyId }) => {
  const [documents, setDocuments] = useState<PropertyDocument[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchDocuments = async () => {
      try {
        const response = await fetch(`http://localhost:8090/api/properties/${propertyId}/documents/public`);
        if (response.ok) {
          const data = await response.json();
          setDocuments(data.documents || []);
        }
      } catch (error) {
        console.error('Error fetching documents:', error);
      } finally {
        setLoading(false);
      }
    };

    fetchDocuments();
  }, [propertyId]);

  // Hide section if no documents or still loading
  if (loading || documents.length === 0) {
    return null;
  }

  return (
    <div className="bg-white border border-gray-200 rounded-lg p-4 mt-4">
      <h4 className="font-semibold text-gray-900 mb-3 flex items-center">
        <FileText className="w-5 h-5 mr-2 text-blue-600" />
        Legal Documents
      </h4>
      
      <div className="space-y-1 mb-4">
        {documents.map((doc) => (
          <div key={doc.id} className="text-sm text-gray-700">
            • {doc.display_name}
          </div>
        ))}
      </div>
      
      <div className="pt-3 border-t border-gray-100">
        <p className="text-xs text-gray-500">
          Documents are provided by the seller/agent for reference only. 
          The platform does not verify or guarantee legal ownership or title.
        </p>
      </div>
    </div>
  );
};