import React from 'react';

interface BankCardProps {
  bank: {
    id: string;
    name: string;
    logo: string;
    minRate: number;
    maxRate: number;
    processingFee: string;
    features: string[];
  };
  isSelected: boolean;
  onSelect: (bankId: string) => void;
}

export const BankCard: React.FC<BankCardProps> = ({ bank, isSelected, onSelect }) => {
  return (
    <div
      onClick={() => onSelect(bank.id)}
      className={`p-6 border-2 rounded-xl cursor-pointer transition-all transform hover:scale-105 ${
        isSelected
          ? 'border-blue-500 bg-blue-50 shadow-lg ring-2 ring-blue-200'
          : 'border-gray-200 hover:border-blue-300 hover:shadow-md bg-white'
      }`}
    >
      <div className="flex items-center gap-4 mb-4">
        <div className="w-12 h-12 bg-gradient-to-br from-blue-100 to-blue-200 rounded-lg flex items-center justify-center">
          <img src={bank.logo} alt={bank.name} className="w-8 h-8 object-contain" />
        </div>
        <div className="flex-1">
          <h3 className="font-bold text-gray-900 text-lg">{bank.name}</h3>
          <p className="text-blue-600 font-semibold">{bank.minRate}% - {bank.maxRate}%</p>
        </div>
        {isSelected && (
          <div className="w-6 h-6 bg-blue-500 rounded-full flex items-center justify-center">
            <span className="text-white text-xs">✓</span>
          </div>
        )}
      </div>
      <div className="space-y-3">
        <div className="bg-gray-50 rounded-lg p-3">
          <p className="text-sm font-medium text-gray-700">Processing Fee</p>
          <p className="text-blue-600 font-semibold">{bank.processingFee}</p>
        </div>
        <div className="flex flex-wrap gap-2">
          {bank.features.slice(0, 2).map((feature, idx) => (
            <span key={idx} className="text-xs bg-blue-100 text-blue-700 px-3 py-1 rounded-full font-medium">
              {feature}
            </span>
          ))}
        </div>
      </div>
    </div>
  );
};