import React from 'react';

interface BankComparison {
  bankId: string;
  bankName: string;
  interestRate: number;
  emi: number;
  totalInterest: number;
  totalAmount: number;
  processingFee: string;
}

interface BankComparisonTableProps {
  comparisons: BankComparison[];
  onSelectBank: (bankId: string) => void;
}

export const BankComparisonTable: React.FC<BankComparisonTableProps> = ({
  comparisons,
  onSelectBank
}) => {
  if (comparisons.length === 0) {
    return (
      <div className="text-center py-8 text-gray-500">
        Select banks to compare loan offers
      </div>
    );
  }

  return (
    <div className="overflow-x-auto">
      <table className="w-full border-collapse bg-white rounded-lg shadow-sm">
        <thead>
          <tr className="bg-gray-50">
            <th className="border border-gray-200 px-4 py-3 text-left font-semibold">Bank</th>
            <th className="border border-gray-200 px-4 py-3 text-center font-semibold">Rate</th>
            <th className="border border-gray-200 px-4 py-3 text-center font-semibold">EMI</th>
            <th className="border border-gray-200 px-4 py-3 text-center font-semibold">Total Interest</th>
            <th className="border border-gray-200 px-4 py-3 text-center font-semibold">Processing Fee</th>
            <th className="border border-gray-200 px-4 py-3 text-center font-semibold">Action</th>
          </tr>
        </thead>
        <tbody>
          {comparisons.map((comparison, index) => (
            <tr key={comparison.bankId} className={index % 2 === 0 ? 'bg-white' : 'bg-gray-50'}>
              <td className="border border-gray-200 px-4 py-3 font-medium">
                {comparison.bankName}
              </td>
              <td className="border border-gray-200 px-4 py-3 text-center">
                {comparison.interestRate}%
              </td>
              <td className="border border-gray-200 px-4 py-3 text-center font-semibold text-blue-600">
                ₹{comparison.emi.toLocaleString('en-IN')}
              </td>
              <td className="border border-gray-200 px-4 py-3 text-center text-green-600">
                ₹{comparison.totalInterest.toLocaleString('en-IN')}
              </td>
              <td className="border border-gray-200 px-4 py-3 text-center text-sm">
                {comparison.processingFee}
              </td>
              <td className="border border-gray-200 px-4 py-3 text-center">
                <button
                  onClick={() => onSelectBank(comparison.bankId)}
                  className="bg-blue-600 text-white px-3 py-1 rounded text-sm hover:bg-blue-700"
                >
                  Select
                </button>
              </td>
            </tr>
          ))}
        </tbody>
      </table>
    </div>
  );
};