import React, { useState } from 'react';
import { BANK_CATEGORIES } from '../../data/banks';

interface Bank {
  id: string;
  name: string;
  logo: string;
  minRate: number;
  maxRate: number;
  processingFee: string;
  features: string[];
}

interface BankDropdownProps {
  selectedBankId: string | null;
  onBankSelect: (bankId: string) => void;
}

export const BankDropdown: React.FC<BankDropdownProps> = ({
  selectedBankId,
  onBankSelect
}) => {
  const [isOpen, setIsOpen] = useState(false);
  const [selectedCategory, setSelectedCategory] = useState<string | null>(null);

  const allBanks = Object.values(BANK_CATEGORIES).flat();
  const selectedBank = allBanks.find(bank => bank.id === selectedBankId);

  const handleCategorySelect = (category: string) => {
    setSelectedCategory(category);
  };

  const handleBankSelect = (bankId: string) => {
    onBankSelect(bankId);
    setIsOpen(false);
    setSelectedCategory(null);
  };

  const handleBack = () => {
    setSelectedCategory(null);
  };

  return (
    <div className="relative">
      <div className="text-center mb-4">
        <h3 className="text-xl font-bold text-gray-900 mb-2">Select Bank</h3>
        <p className="text-gray-600">Choose from 46+ banks and housing finance companies</p>
      </div>

      <button
        onClick={() => setIsOpen(!isOpen)}
        className="w-full p-4 border-2 border-gray-300 rounded-lg bg-white hover:border-blue-400 transition-colors flex items-center justify-between shadow-sm"
        style={{ minHeight: '60px' }}
      >
        <div className="flex items-center gap-3">
          {selectedBank ? (
            <>
              <div className="w-10 h-10 bg-gradient-to-br from-blue-100 to-blue-200 rounded-lg flex items-center justify-center">
                <svg className="w-6 h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-4m-5 0H3m2 0h3M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4" />
                </svg>
              </div>
              <div className="text-left">
                <p className="font-semibold text-gray-900">{selectedBank.name}</p>
                <p className="text-sm text-blue-600">{selectedBank.minRate}% - {selectedBank.maxRate}%</p>
              </div>
            </>
          ) : (
            <div className="flex items-center gap-3">
              <div className="w-10 h-10 bg-gray-100 rounded-lg flex items-center justify-center">
                <svg className="w-6 h-6 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-4m-5 0H3m2 0h3M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4" />
                </svg>
              </div>
              <div>
                <p className="font-medium text-gray-500">Select a bank</p>
                <p className="text-sm text-gray-400">Choose from 46+ options</p>
              </div>
            </div>
          )}
        </div>
        <svg className={`w-5 h-5 transition-transform ${isOpen ? 'rotate-180' : ''}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
        </svg>
      </button>

      {isOpen && (
        <div className="absolute top-full left-0 right-0 mt-2 bg-white border-2 border-gray-200 rounded-lg shadow-lg z-50 max-h-48 overflow-hidden">
          {!selectedCategory ? (
            // Category Selection
            <div>
              <div className="p-3 bg-blue-50 border-b text-center">
                <p className="text-sm font-semibold text-blue-800">Select Bank Type</p>
              </div>
              <div className="p-2">
              {Object.entries(BANK_CATEGORIES).map(([category, banks]) => (
                <button
                  key={category}
                  onClick={() => handleCategorySelect(category)}
                  className="w-full p-2 hover:bg-blue-50 transition-colors flex items-center justify-between text-left"
                >
                  <div>
                    <p className="text-sm font-medium text-gray-900">{category}</p>
                    <p className="text-xs text-gray-500">{banks.length} options</p>
                  </div>
                  <svg className="w-4 h-4 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
                  </svg>
                </button>
              ))}
              </div>
            </div>
          ) : (
            // Bank Selection within Category
            <div>
              <div className="p-3 border-b bg-gray-50 flex items-center gap-2">
                <button onClick={handleBack} className="text-blue-600 hover:text-blue-700">
                  <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
                  </svg>
                </button>
                <span className="font-medium text-gray-700">{selectedCategory}</span>
              </div>
              <div className="overflow-y-auto max-h-36">
                {BANK_CATEGORIES[selectedCategory]?.map((bank) => (
                  <button
                    key={bank.id}
                    onClick={() => handleBankSelect(bank.id)}
                    className="w-full p-2 hover:bg-blue-50 transition-colors flex items-center gap-2 text-left border-b border-gray-50 last:border-b-0"
                  >
                    <div className="w-6 h-6 bg-gradient-to-br from-blue-100 to-blue-200 rounded flex items-center justify-center flex-shrink-0">
                      <svg className="w-4 h-4 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-4m-5 0H3m2 0h3M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4" />
                      </svg>
                    </div>
                    <div className="flex-1 min-w-0">
                      <p className="text-sm font-medium text-gray-900 truncate">{bank.name}</p>
                      <p className="text-xs text-blue-600">{bank.minRate}% - {bank.maxRate}%</p>
                    </div>
                  </button>
                ))}
              </div>
            </div>
          )}
        </div>
      )}
    </div>
  );
};