import React, { useState } from 'react';
import { getBankRate, BANKS_DATA } from '../../data/banks';

interface MiniEMICalculatorProps {
  propertyPrice: number;
  onOpenFullCalculator?: () => void;
}

export const MiniEMICalculator: React.FC<MiniEMICalculatorProps> = ({
  propertyPrice,
  onOpenFullCalculator
}) => {
  const [downPayment, setDownPayment] = useState(propertyPrice * 0.2); // 20% default
  const [tenure, setTenure] = useState(20);
  
  const loanAmount = propertyPrice - downPayment;
  const defaultRate = 8.5; // Default rate
  
  const calculateEMI = (principal: number, rate: number, years: number) => {
    const monthlyRate = rate / 12 / 100;
    const numPayments = years * 12;
    
    const emi = (principal * monthlyRate * Math.pow(1 + monthlyRate, numPayments)) / 
                (Math.pow(1 + monthlyRate, numPayments) - 1);
    
    return Math.round(emi);
  };
  
  const emi = calculateEMI(loanAmount, defaultRate, tenure);
  
  return (
    <div className="bg-gradient-to-r from-blue-50 to-indigo-50 border border-blue-200 rounded-lg p-4">
      <div className="flex items-center justify-between mb-3">
        <h3 className="font-semibold text-gray-900">EMI Calculator</h3>
        <span className="text-xs bg-blue-100 text-blue-700 px-2 py-1 rounded">
          @ {defaultRate}% rate
        </span>
      </div>
      
      <div className="space-y-3">
        <div>
          <label className="block text-xs text-gray-600 mb-1">
            Down Payment: ₹{(downPayment/100000).toFixed(1)}L
          </label>
          <input
            type="range"
            min={propertyPrice * 0.1}
            max={propertyPrice * 0.5}
            step={50000}
            value={downPayment}
            onChange={(e) => setDownPayment(parseInt(e.target.value))}
            className="w-full h-1 bg-gray-200 rounded-lg appearance-none cursor-pointer"
          />
        </div>
        
        <div>
          <label className="block text-xs text-gray-600 mb-1">
            Tenure: {tenure} years
          </label>
          <input
            type="range"
            min={5}
            max={30}
            step={5}
            value={tenure}
            onChange={(e) => setTenure(parseInt(e.target.value))}
            className="w-full h-1 bg-gray-200 rounded-lg appearance-none cursor-pointer"
          />
        </div>
        
        <div className="bg-white rounded-lg p-3 text-center">
          <p className="text-xs text-gray-600">Monthly EMI</p>
          <p className="text-xl font-bold text-blue-600">
            ₹{emi.toLocaleString('en-IN')}
          </p>
          <p className="text-xs text-gray-500">
            Loan: ₹{(loanAmount/100000).toFixed(1)}L
          </p>
        </div>
        
        <button
          onClick={onOpenFullCalculator}
          className="w-full bg-blue-600 hover:bg-blue-700 text-white py-2 px-3 rounded-md text-sm font-medium transition-colors"
        >
          🏦 Compare Bank Rates
        </button>
      </div>
    </div>
  );
};