import React, { useState, useEffect } from 'react';
import { TrendingUp, Shield, AlertCircle, CheckCircle, Info } from 'lucide-react';

interface Insight {
  id: number;
  market_name: string;
  city: string;
  recommendation: 'Buy' | 'Watch' | 'Avoid';
  confidence: number;
  reasoning: string[];
  risk_factors: string[];
  opportunities: string[];
  price_forecast: {
    next_6_months: number;
    next_12_months: number;
  };
}

export function InvestmentInsights() {
  const [insights, setInsights] = useState<Insight[]>([]);
  const [selectedCity, setSelectedCity] = useState('all');

  useEffect(() => {
    const mockInsights: Insight[] = [
      {
        id: 1,
        market_name: "Bandra West",
        city: "Mumbai",
        recommendation: "Buy",
        confidence: 85,
        reasoning: [
          "Strong demand from IT professionals",
          "Limited supply due to land constraints",
          "Excellent connectivity to business districts",
          "Premium location with lifestyle amenities"
        ],
        risk_factors: [
          "High entry prices may limit buyer pool",
          "Monsoon flooding concerns in some areas"
        ],
        opportunities: [
          "Upcoming metro connectivity",
          "Commercial development in nearby areas"
        ],
        price_forecast: {
          next_6_months: 8.5,
          next_12_months: 15.2
        }
      },
      {
        id: 2,
        market_name: "Whitefield",
        city: "Bangalore",
        recommendation: "Watch",
        confidence: 72,
        reasoning: [
          "Growing IT hub with major companies",
          "Infrastructure development in progress",
          "Relatively affordable compared to central areas"
        ],
        risk_factors: [
          "Traffic congestion during peak hours",
          "Dependency on IT sector growth",
          "Water scarcity issues"
        ],
        opportunities: [
          "Proposed metro extension",
          "New residential projects launching"
        ],
        price_forecast: {
          next_6_months: 5.2,
          next_12_months: 12.8
        }
      }
    ];
    setInsights(mockInsights);
  }, [selectedCity]);

  const getRecommendationColor = (rec: string) => {
    switch (rec) {
      case 'Buy': return 'bg-green-500';
      case 'Watch': return 'bg-yellow-500';
      case 'Avoid': return 'bg-red-500';
      default: return 'bg-gray-500';
    }
  };

  const getRecommendationIcon = (rec: string) => {
    switch (rec) {
      case 'Buy': return <CheckCircle className="w-5 h-5 text-white" />;
      case 'Watch': return <Info className="w-5 h-5 text-white" />;
      case 'Avoid': return <AlertCircle className="w-5 h-5 text-white" />;
      default: return <Info className="w-5 h-5 text-white" />;
    }
  };

  return (
    <div className="bg-white rounded-lg shadow-sm">
      <div className="p-6 border-b border-gray-200">
        <div className="flex justify-between items-center">
          <h2 className="text-xl font-semibold text-gray-900">Investment Insights & Recommendations</h2>
          <select
            value={selectedCity}
            onChange={(e) => setSelectedCity(e.target.value)}
            className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
          >
            <option value="all">All Cities</option>
            <option value="Mumbai">Mumbai</option>
            <option value="Delhi">Delhi</option>
            <option value="Bangalore">Bangalore</option>
            <option value="Pune">Pune</option>
          </select>
        </div>
      </div>

      <div className="p-6">
        <div className="space-y-6">
          {insights.map((insight) => (
            <div key={insight.id} className="border border-gray-200 rounded-lg p-6">
              {/* Header */}
              <div className="flex justify-between items-start mb-6">
                <div>
                  <h3 className="text-lg font-semibold text-gray-900">{insight.market_name}</h3>
                  <p className="text-gray-600">{insight.city}</p>
                </div>
                <div className="flex items-center gap-4">
                  <div className="text-center">
                    <div className="text-sm text-gray-500">Confidence</div>
                    <div className="text-xl font-bold text-blue-600">{insight.confidence}%</div>
                  </div>
                  <div className={`flex items-center gap-2 px-4 py-2 rounded-lg ${getRecommendationColor(insight.recommendation)}`}>
                    {getRecommendationIcon(insight.recommendation)}
                    <span className="text-white font-semibold">{insight.recommendation}</span>
                  </div>
                </div>
              </div>

              {/* Price Forecast */}
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                <div className="bg-blue-50 p-4 rounded-lg">
                  <h4 className="font-semibold text-blue-900 mb-2">Price Forecast</h4>
                  <div className="space-y-2">
                    <div className="flex justify-between">
                      <span className="text-blue-700">6 months:</span>
                      <span className="font-semibold text-green-600">+{insight.price_forecast.next_6_months}%</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-blue-700">12 months:</span>
                      <span className="font-semibold text-green-600">+{insight.price_forecast.next_12_months}%</span>
                    </div>
                  </div>
                </div>

                <div className="bg-gray-50 p-4 rounded-lg">
                  <h4 className="font-semibold text-gray-900 mb-2">Key Metrics</h4>
                  <div className="space-y-2 text-sm">
                    <div className="flex justify-between">
                      <span className="text-gray-600">Investment Grade:</span>
                      <span className="font-semibold">A+</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-gray-600">Liquidity:</span>
                      <span className="font-semibold">High</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-gray-600">Risk Level:</span>
                      <span className="font-semibold">Low</span>
                    </div>
                  </div>
                </div>
              </div>

              {/* Analysis Sections */}
              <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                {/* Reasoning */}
                <div>
                  <h4 className="font-semibold text-gray-900 mb-3 flex items-center gap-2">
                    <CheckCircle className="w-4 h-4 text-green-500" />
                    Why {insight.recommendation}
                  </h4>
                  <ul className="space-y-2">
                    {insight.reasoning.map((reason, index) => (
                      <li key={index} className="text-sm text-gray-600 flex items-start gap-2">
                        <div className="w-1.5 h-1.5 bg-green-500 rounded-full mt-2 flex-shrink-0"></div>
                        {reason}
                      </li>
                    ))}
                  </ul>
                </div>

                {/* Risk Factors */}
                <div>
                  <h4 className="font-semibold text-gray-900 mb-3 flex items-center gap-2">
                    <Shield className="w-4 h-4 text-red-500" />
                    Risk Factors
                  </h4>
                  <ul className="space-y-2">
                    {insight.risk_factors.map((risk, index) => (
                      <li key={index} className="text-sm text-gray-600 flex items-start gap-2">
                        <div className="w-1.5 h-1.5 bg-red-500 rounded-full mt-2 flex-shrink-0"></div>
                        {risk}
                      </li>
                    ))}
                  </ul>
                </div>

                {/* Opportunities */}
                <div>
                  <h4 className="font-semibold text-gray-900 mb-3 flex items-center gap-2">
                    <TrendingUp className="w-4 h-4 text-blue-500" />
                    Opportunities
                  </h4>
                  <ul className="space-y-2">
                    {insight.opportunities.map((opportunity, index) => (
                      <li key={index} className="text-sm text-gray-600 flex items-start gap-2">
                        <div className="w-1.5 h-1.5 bg-blue-500 rounded-full mt-2 flex-shrink-0"></div>
                        {opportunity}
                      </li>
                    ))}
                  </ul>
                </div>
              </div>

              {/* Disclaimer */}
              <div className="mt-6 p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
                <p className="text-xs text-yellow-800">
                  <strong>Disclaimer:</strong> These recommendations are based on data analysis and market trends. 
                  Past performance does not guarantee future results. Please consult with financial advisors before making investment decisions.
                </p>
              </div>
            </div>
          ))}
        </div>
      </div>
    </div>
  );
}