import React, { useState, useEffect, useRef } from 'react';
import { Filter, Search, MapPin } from 'lucide-react';

interface MicroMarket {
  id: number;
  name: string;
  city: string;
  lat: number;
  lng: number;
  overall_score: number;
  tier: string;
  growth_percentage: number;
  rental_yield: number;
  risk_level: string;
}

export function MicroMarketHeatmap() {
  const [markets, setMarkets] = useState<MicroMarket[]>([]);
  const [selectedCity, setSelectedCity] = useState('Mumbai');
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedMarket, setSelectedMarket] = useState<MicroMarket | null>(null);
  const mapRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    // Mock data for heatmap
    const mockMarkets: MicroMarket[] = [
      {
        id: 1,
        name: "Bandra West",
        city: "Mumbai",
        lat: 19.0596,
        lng: 72.8295,
        overall_score: 92,
        tier: "High Performing",
        growth_percentage: 15.2,
        rental_yield: 3.2,
        risk_level: "Low"
      },
      {
        id: 2,
        name: "Powai",
        city: "Mumbai", 
        lat: 19.1176,
        lng: 72.9060,
        overall_score: 82,
        tier: "High Performing",
        growth_percentage: 11.8,
        rental_yield: 3.8,
        risk_level: "Low"
      },
      {
        id: 3,
        name: "Andheri East",
        city: "Mumbai",
        lat: 19.1136,
        lng: 72.8697,
        overall_score: 75,
        tier: "Emerging",
        growth_percentage: 9.5,
        rental_yield: 4.1,
        risk_level: "Medium"
      }
    ];
    setMarkets(mockMarkets);
  }, [selectedCity]);

  const getScoreColor = (score: number) => {
    if (score >= 80) return '#10B981'; // Green
    if (score >= 65) return '#3B82F6'; // Blue  
    if (score >= 50) return '#F59E0B'; // Yellow
    return '#EF4444'; // Red
  };

  const getRiskColor = (risk: string) => {
    switch (risk) {
      case 'Low': return '#10B981';
      case 'Medium': return '#F59E0B';
      case 'High': return '#EF4444';
      default: return '#6B7280';
    }
  };

  return (
    <div className="bg-white rounded-lg shadow-sm">
      {/* Controls */}
      <div className="p-6 border-b border-gray-200">
        <div className="flex flex-col md:flex-row gap-4">
          <div className="flex-1">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
              <input
                type="text"
                placeholder="Search micro-markets..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
              />
            </div>
          </div>
          <select
            value={selectedCity}
            onChange={(e) => setSelectedCity(e.target.value)}
            className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
          >
            <option value="Mumbai">Mumbai</option>
            <option value="Delhi">Delhi</option>
            <option value="Bangalore">Bangalore</option>
            <option value="Pune">Pune</option>
          </select>
        </div>
      </div>

      {/* Map Container */}
      <div className="relative">
        <div
          ref={mapRef}
          className="h-96 bg-gray-100 flex items-center justify-center"
        >
          <div className="text-center">
            <MapPin className="w-12 h-12 text-gray-400 mx-auto mb-2" />
            <p className="text-gray-500">Interactive map will load here</p>
            <p className="text-sm text-gray-400">Showing {markets.length} micro-markets in {selectedCity}</p>
          </div>
        </div>

        {/* Legend */}
        <div className="absolute top-4 right-4 bg-white p-4 rounded-lg shadow-md">
          <h4 className="font-semibold text-sm mb-3">Performance Score</h4>
          <div className="space-y-2">
            <div className="flex items-center gap-2">
              <div className="w-4 h-4 rounded-full bg-green-500"></div>
              <span className="text-xs">High (80-100)</span>
            </div>
            <div className="flex items-center gap-2">
              <div className="w-4 h-4 rounded-full bg-blue-500"></div>
              <span className="text-xs">Emerging (65-79)</span>
            </div>
            <div className="flex items-center gap-2">
              <div className="w-4 h-4 rounded-full bg-yellow-500"></div>
              <span className="text-xs">Stable (50-64)</span>
            </div>
            <div className="flex items-center gap-2">
              <div className="w-4 h-4 rounded-full bg-red-500"></div>
              <span className="text-xs">Underperforming (&lt;50)</span>
            </div>
          </div>
        </div>
      </div>

      {/* Market Cards Grid */}
      <div className="p-6">
        <h3 className="text-lg font-semibold mb-4">Micro-Markets in {selectedCity}</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
          {markets
            .filter(market => 
              market.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
              market.city.toLowerCase().includes(searchTerm.toLowerCase())
            )
            .map((market) => (
            <div
              key={market.id}
              className="border border-gray-200 rounded-lg p-4 hover:shadow-md transition-shadow cursor-pointer"
              onClick={() => setSelectedMarket(market)}
            >
              <div className="flex justify-between items-start mb-3">
                <h4 className="font-semibold text-gray-900">{market.name}</h4>
                <div 
                  className="w-3 h-3 rounded-full"
                  style={{ backgroundColor: getScoreColor(market.overall_score) }}
                ></div>
              </div>
              
              <div className="space-y-2 text-sm">
                <div className="flex justify-between">
                  <span className="text-gray-600">Score:</span>
                  <span className="font-medium">{market.overall_score}/100</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-600">Growth:</span>
                  <span className="font-medium text-green-600">+{market.growth_percentage}%</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-600">Rental Yield:</span>
                  <span className="font-medium">{market.rental_yield}%</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-600">Risk:</span>
                  <span 
                    className="font-medium"
                    style={{ color: getRiskColor(market.risk_level) }}
                  >
                    {market.risk_level}
                  </span>
                </div>
              </div>
            </div>
          ))}
        </div>
      </div>
    </div>
  );
}