import React, { useState, useEffect } from 'react';
import { Trophy, TrendingUp, ArrowUp, ArrowDown, Eye, Star } from 'lucide-react';

interface MarketRanking {
  id: number;
  rank: number;
  name: string;
  city: string;
  overall_score: number;
  growth_percentage: number;
  tier: string;
  recommendation: 'Buy' | 'Watch' | 'Avoid';
  price_trend: 'up' | 'down' | 'stable';
  avg_price_per_sqft: number;
}

export function MicroMarketLeaderboard() {
  const [rankings, setRankings] = useState<MarketRanking[]>([]);
  const [selectedCity, setSelectedCity] = useState('all');

  useEffect(() => {
    const mockRankings: MarketRanking[] = [
      {
        id: 1,
        rank: 1,
        name: "Bandra West",
        city: "Mumbai",
        overall_score: 92,
        growth_percentage: 15.2,
        tier: "High Performing",
        recommendation: "Buy",
        price_trend: "up",
        avg_price_per_sqft: 45000
      },
      {
        id: 2,
        rank: 2,
        name: "Koramangala",
        city: "Bangalore",
        overall_score: 89,
        growth_percentage: 18.7,
        tier: "High Performing",
        recommendation: "Buy",
        price_trend: "up",
        avg_price_per_sqft: 12500
      },
      {
        id: 3,
        rank: 3,
        name: "Gurgaon Sector 54",
        city: "Delhi",
        overall_score: 85,
        growth_percentage: 12.4,
        tier: "High Performing",
        recommendation: "Watch",
        price_trend: "up",
        avg_price_per_sqft: 8500
      },
      {
        id: 4,
        rank: 4,
        name: "Powai",
        city: "Mumbai",
        overall_score: 82,
        growth_percentage: 11.8,
        tier: "High Performing",
        recommendation: "Buy",
        price_trend: "stable",
        avg_price_per_sqft: 18500
      },
      {
        id: 5,
        rank: 5,
        name: "Whitefield",
        city: "Bangalore",
        overall_score: 78,
        growth_percentage: 14.1,
        tier: "Emerging",
        recommendation: "Watch",
        price_trend: "up",
        avg_price_per_sqft: 9200
      }
    ];
    setRankings(mockRankings);
  }, [selectedCity]);

  const getRecommendationColor = (rec: string) => {
    switch (rec) {
      case 'Buy': return 'bg-green-100 text-green-800';
      case 'Watch': return 'bg-yellow-100 text-yellow-800';
      case 'Avoid': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getRankIcon = (rank: number) => {
    if (rank === 1) return <Trophy className="w-5 h-5 text-yellow-500" />;
    if (rank === 2) return <Trophy className="w-5 h-5 text-gray-400" />;
    if (rank === 3) return <Trophy className="w-5 h-5 text-amber-600" />;
    return <span className="w-5 h-5 flex items-center justify-center text-sm font-bold text-gray-600">#{rank}</span>;
  };

  return (
    <div className="bg-white rounded-lg shadow-sm">
      <div className="p-6 border-b border-gray-200">
        <div className="flex justify-between items-center">
          <h2 className="text-xl font-semibold text-gray-900">Top Performing Micro-Markets</h2>
          <select
            value={selectedCity}
            onChange={(e) => setSelectedCity(e.target.value)}
            className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
          >
            <option value="all">All Cities</option>
            <option value="Mumbai">Mumbai</option>
            <option value="Delhi">Delhi</option>
            <option value="Bangalore">Bangalore</option>
            <option value="Pune">Pune</option>
          </select>
        </div>
      </div>

      <div className="p-6">
        <div className="space-y-4">
          {rankings.map((market) => (
            <div
              key={market.id}
              className="flex items-center justify-between p-4 border border-gray-200 rounded-lg hover:shadow-md transition-shadow"
            >
              <div className="flex items-center gap-4">
                <div className="flex items-center justify-center w-10 h-10">
                  {getRankIcon(market.rank)}
                </div>
                
                <div>
                  <h3 className="font-semibold text-gray-900">{market.name}</h3>
                  <p className="text-sm text-gray-600">{market.city}</p>
                </div>
              </div>

              <div className="flex items-center gap-6">
                <div className="text-center">
                  <div className="text-2xl font-bold text-blue-600">{market.overall_score}</div>
                  <div className="text-xs text-gray-500">Score</div>
                </div>

                <div className="text-center">
                  <div className="flex items-center gap-1">
                    {market.price_trend === 'up' ? (
                      <ArrowUp className="w-4 h-4 text-green-500" />
                    ) : market.price_trend === 'down' ? (
                      <ArrowDown className="w-4 h-4 text-red-500" />
                    ) : (
                      <div className="w-4 h-4" />
                    )}
                    <span className="font-semibold text-green-600">+{market.growth_percentage}%</span>
                  </div>
                  <div className="text-xs text-gray-500">Growth</div>
                </div>

                <div className="text-center">
                  <div className="font-semibold">₹{market.avg_price_per_sqft.toLocaleString()}</div>
                  <div className="text-xs text-gray-500">per sq ft</div>
                </div>

                <div className="text-center">
                  <span className={`px-3 py-1 rounded-full text-xs font-medium ${getRecommendationColor(market.recommendation)}`}>
                    {market.recommendation}
                  </span>
                </div>

                <div className="flex gap-2">
                  <button className="p-2 text-gray-400 hover:text-blue-600 rounded-lg hover:bg-blue-50">
                    <Eye className="w-4 h-4" />
                  </button>
                  <button className="p-2 text-gray-400 hover:text-yellow-600 rounded-lg hover:bg-yellow-50">
                    <Star className="w-4 h-4" />
                  </button>
                </div>
              </div>
            </div>
          ))}
        </div>
      </div>
    </div>
  );
}