import React, { useState, useEffect } from 'react';
import { Bell, Plus, Trash2, TrendingUp, AlertTriangle } from 'lucide-react';

interface WatchlistItem {
  id: number;
  name: string;
  city: string;
  overall_score: number;
  alerts: Alert[];
  last_updated: string;
}

interface Alert {
  id: number;
  type: 'demand_spike' | 'supply_drop' | 'price_change' | 'infrastructure';
  message: string;
  severity: 'low' | 'medium' | 'high';
  timestamp: string;
}

export function MicroMarketWatchlist() {
  const [watchlist, setWatchlist] = useState<WatchlistItem[]>([]);
  const [showAddModal, setShowAddModal] = useState(false);

  useEffect(() => {
    const mockWatchlist: WatchlistItem[] = [
      {
        id: 1,
        name: "Bandra West",
        city: "Mumbai",
        overall_score: 92,
        alerts: [
          {
            id: 1,
            type: 'demand_spike',
            message: 'Demand increased by 25% in last 30 days',
            severity: 'high',
            timestamp: '2024-01-15T10:30:00Z'
          }
        ],
        last_updated: '2024-01-15T10:30:00Z'
      },
      {
        id: 2,
        name: "Whitefield",
        city: "Bangalore",
        overall_score: 78,
        alerts: [
          {
            id: 2,
            type: 'infrastructure',
            message: 'New metro line construction announced',
            severity: 'medium',
            timestamp: '2024-01-14T15:20:00Z'
          }
        ],
        last_updated: '2024-01-14T15:20:00Z'
      }
    ];
    setWatchlist(mockWatchlist);
  }, []);

  const getSeverityColor = (severity: string) => {
    switch (severity) {
      case 'high': return 'text-red-600 bg-red-50';
      case 'medium': return 'text-yellow-600 bg-yellow-50';
      case 'low': return 'text-green-600 bg-green-50';
      default: return 'text-gray-600 bg-gray-50';
    }
  };

  const getAlertIcon = (type: string) => {
    switch (type) {
      case 'demand_spike': return <TrendingUp className="w-4 h-4" />;
      case 'supply_drop': return <AlertTriangle className="w-4 h-4" />;
      case 'price_change': return <TrendingUp className="w-4 h-4" />;
      case 'infrastructure': return <Bell className="w-4 h-4" />;
      default: return <Bell className="w-4 h-4" />;
    }
  };

  const removeFromWatchlist = (id: number) => {
    setWatchlist(prev => prev.filter(item => item.id !== id));
  };

  return (
    <div className="bg-white rounded-lg shadow-sm">
      <div className="p-6 border-b border-gray-200">
        <div className="flex justify-between items-center">
          <h2 className="text-xl font-semibold text-gray-900">My Watchlist</h2>
          <button
            onClick={() => setShowAddModal(true)}
            className="flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
          >
            <Plus className="w-4 h-4" />
            Add Market
          </button>
        </div>
      </div>

      <div className="p-6">
        {watchlist.length === 0 ? (
          <div className="text-center py-12">
            <Bell className="w-12 h-12 text-gray-300 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">No markets in watchlist</h3>
            <p className="text-gray-500 mb-4">Add micro-markets to get alerts on market changes</p>
            <button
              onClick={() => setShowAddModal(true)}
              className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
            >
              Add Your First Market
            </button>
          </div>
        ) : (
          <div className="space-y-4">
            {watchlist.map((item) => (
              <div key={item.id} className="border border-gray-200 rounded-lg p-4">
                <div className="flex justify-between items-start mb-4">
                  <div>
                    <h3 className="font-semibold text-gray-900">{item.name}</h3>
                    <p className="text-sm text-gray-600">{item.city}</p>
                  </div>
                  <div className="flex items-center gap-2">
                    <div className="text-right">
                      <div className="text-lg font-bold text-blue-600">{item.overall_score}</div>
                      <div className="text-xs text-gray-500">Score</div>
                    </div>
                    <button
                      onClick={() => removeFromWatchlist(item.id)}
                      className="p-2 text-gray-400 hover:text-red-600 rounded-lg hover:bg-red-50"
                    >
                      <Trash2 className="w-4 h-4" />
                    </button>
                  </div>
                </div>

                {item.alerts.length > 0 && (
                  <div className="space-y-2">
                    <h4 className="text-sm font-medium text-gray-700">Recent Alerts</h4>
                    {item.alerts.map((alert) => (
                      <div
                        key={alert.id}
                        className={`flex items-start gap-3 p-3 rounded-lg ${getSeverityColor(alert.severity)}`}
                      >
                        {getAlertIcon(alert.type)}
                        <div className="flex-1">
                          <p className="text-sm font-medium">{alert.message}</p>
                          <p className="text-xs opacity-75">
                            {new Date(alert.timestamp).toLocaleDateString()}
                          </p>
                        </div>
                      </div>
                    ))}
                  </div>
                )}

                <div className="mt-4 pt-4 border-t border-gray-100">
                  <p className="text-xs text-gray-500">
                    Last updated: {new Date(item.last_updated).toLocaleString()}
                  </p>
                </div>
              </div>
            ))}
          </div>
        )}
      </div>
    </div>
  );
}