import React from 'react';
import { AIInsight } from '../../types/negotiation';

interface AISuggestionBoxProps {
  aiInsight: AIInsight | null;
}

const AISuggestionBox: React.FC<AISuggestionBoxProps> = ({ aiInsight }) => {
  if (!aiInsight) {
    return null;
  }

  const formatAmount = (amount: number) => {
    return new Intl.NumberFormat('en-IN', {
      style: 'currency',
      currency: 'INR',
      maximumFractionDigits: 0,
    }).format(amount);
  };

  const getClassificationColor = (classification: string) => {
    switch (classification) {
      case 'LOWBALL':
        return 'text-red-600 bg-red-50';
      case 'FAIR':
        return 'text-green-600 bg-green-50';
      case 'AGGRESSIVE':
        return 'text-blue-600 bg-blue-50';
      default:
        return 'text-gray-600 bg-gray-50';
    }
  };

  const getClassificationText = (classification: string) => {
    switch (classification) {
      case 'LOWBALL':
        return 'Below Market Value';
      case 'FAIR':
        return 'Fair Market Value';
      case 'AGGRESSIVE':
        return 'Above Market Value';
      default:
        return classification;
    }
  };

  return (
    <div className="bg-white rounded-lg shadow-sm border p-6">
      <div className="flex items-center mb-4">
        <div className="w-8 h-8 bg-purple-100 rounded-full flex items-center justify-center mr-3">
          <svg className="w-4 h-4 text-purple-600" fill="currentColor" viewBox="0 0 20 20">
            <path fillRule="evenodd" d="M11.49 3.17c-.38-1.56-2.6-1.56-2.98 0a1.532 1.532 0 01-2.286.948c-1.372-.836-2.942.734-2.106 2.106.54.886.061 2.042-.947 2.287-1.561.379-1.561 2.6 0 2.978a1.532 1.532 0 01.947 2.287c-.836 1.372.734 2.942 2.106 2.106a1.532 1.532 0 012.287.947c.379 1.561 2.6 1.561 2.978 0a1.533 1.533 0 012.287-.947c1.372.836 2.942-.734 2.106-2.106a1.533 1.533 0 01.947-2.287c1.561-.379 1.561-2.6 0-2.978a1.532 1.532 0 01-.947-2.287c.836-1.372-.734-2.942-2.106-2.106a1.532 1.532 0 01-2.287-.947zM10 13a3 3 0 100-6 3 3 0 000 6z" clipRule="evenodd" />
          </svg>
        </div>
        <h3 className="text-lg font-medium text-gray-900">AI Market Analysis</h3>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
        <div className="bg-gray-50 p-3 rounded-lg">
          <p className="text-sm text-gray-600 mb-1">Market Price</p>
          <p className="text-lg font-semibold text-gray-900">
            {formatAmount(aiInsight.market_price)}
          </p>
        </div>
        
        <div className={`p-3 rounded-lg ${getClassificationColor(aiInsight.classification)}`}>
          <p className="text-sm opacity-75 mb-1">Classification</p>
          <p className="text-lg font-semibold">
            {getClassificationText(aiInsight.classification)}
          </p>
        </div>
      </div>

      <div className="bg-gray-50 p-4 rounded-lg">
        <h4 className="text-sm font-medium text-gray-900 mb-3">Suggested Price Range</h4>
        <div className="grid grid-cols-3 gap-3">
          <div className="text-center">
            <p className="text-xs text-gray-600 mb-1">Low</p>
            <p className="text-sm font-medium text-gray-900">
              {formatAmount(aiInsight.suggested_range.low)}
            </p>
          </div>
          <div className="text-center">
            <p className="text-xs text-gray-600 mb-1">Fair</p>
            <p className="text-sm font-medium text-green-600">
              {formatAmount(aiInsight.suggested_range.fair)}
            </p>
          </div>
          <div className="text-center">
            <p className="text-xs text-gray-600 mb-1">High</p>
            <p className="text-sm font-medium text-gray-900">
              {formatAmount(aiInsight.suggested_range.high)}
            </p>
          </div>
        </div>
      </div>

      <div className="mt-4 p-3 bg-blue-50 rounded-lg">
        <p className="text-xs text-blue-700">
          💡 This analysis is based on market data and comparable properties. 
          Use it as guidance for your negotiation strategy.
        </p>
      </div>
    </div>
  );
};

export default AISuggestionBox;