import React, { useRef, useEffect } from 'react';
import { useWebRTCCall } from '../../hooks/useWebRTCCall';

interface AudioCallProps {
  negotiationId: string;
  userId: string;
  canMakeCall: boolean;
  disabled?: boolean;
}

const AudioCall: React.FC<AudioCallProps> = ({ 
  negotiationId, 
  userId, 
  canMakeCall, 
  disabled = false 
}) => {
  const {
    callState,
    startCall,
    acceptCall,
    rejectCall,
    endCall,
    toggleAudio,
    remoteAudio,
  } = useWebRTCCall(negotiationId, userId);

  const audioRef = useRef<HTMLAudioElement>(null);

  // Connect remote audio element
  useEffect(() => {
    if (audioRef.current) {
      remoteAudio.current = audioRef.current;
    }
  }, [remoteAudio]);

  const formatDuration = (seconds: number) => {
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins}:${secs.toString().padStart(2, '0')}`;
  };

  const getStatusText = () => {
    switch (callState.status) {
      case 'calling':
        return 'Calling...';
      case 'ringing':
        return 'Incoming call';
      case 'connected':
        return `Connected - ${formatDuration(callState.duration)}`;
      case 'ended':
        return 'Call ended';
      case 'rejected':
        return 'Call rejected';
      default:
        return 'Audio Call';
    }
  };

  const getStatusColor = () => {
    switch (callState.status) {
      case 'calling':
        return 'text-yellow-600 bg-yellow-50';
      case 'ringing':
        return 'text-blue-600 bg-blue-50';
      case 'connected':
        return 'text-green-600 bg-green-50';
      case 'ended':
      case 'rejected':
        return 'text-gray-600 bg-gray-50';
      default:
        return 'text-gray-700 bg-white';
    }
  };

  if (!canMakeCall) {
    return (
      <div className="bg-gray-50 rounded-lg border p-4">
        <div className="flex items-center justify-between">
          <div className="flex items-center">
            <div className="w-8 h-8 bg-gray-200 rounded-full flex items-center justify-center mr-3">
              <svg className="w-4 h-4 text-gray-500" fill="currentColor" viewBox="0 0 20 20">
                <path d="M2 3a1 1 0 011-1h2.153a1 1 0 01.986.836l.74 4.435a1 1 0 01-.54 1.06l-1.548.773a11.037 11.037 0 006.105 6.105l.774-1.548a1 1 0 011.059-.54l4.435.74a1 1 0 01.836.986V17a1 1 0 01-1 1h-2C7.82 18 2 12.18 2 5V3z" />
              </svg>
            </div>
            <span className="text-sm text-gray-600">Audio calls not available</span>
          </div>
        </div>
        <p className="text-xs text-gray-500 mt-2">
          Calls are only available during active negotiations
        </p>
      </div>
    );
  }

  return (
    <div className="bg-white rounded-lg border p-4">
      {/* Hidden audio element for remote stream */}
      <audio ref={audioRef} autoPlay playsInline volume={1.0} />
      
      <div className="flex items-center justify-between mb-3">
        <div className="flex items-center">
          <div className="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center mr-3">
            <svg className="w-4 h-4 text-blue-600" fill="currentColor" viewBox="0 0 20 20">
              <path d="M2 3a1 1 0 011-1h2.153a1 1 0 01.986.836l.74 4.435a1 1 0 01-.54 1.06l-1.548.773a11.037 11.037 0 006.105 6.105l.774-1.548a1 1 0 011.059-.54l4.435.74a1 1 0 01.836.986V17a1 1 0 01-1 1h-2C7.82 18 2 12.18 2 5V3z" />
            </svg>
          </div>
          <div>
            <h4 className="font-medium text-gray-900">Audio Call</h4>
            <p className={`text-sm px-2 py-1 rounded-full ${getStatusColor()}`}>
              {getStatusText()}
            </p>
          </div>
        </div>
      </div>

      {/* Error message */}
      {callState.error && (
        <div className="mb-3 p-2 bg-red-50 border border-red-200 rounded text-sm text-red-600">
          {callState.error}
        </div>
      )}

      {/* Call controls */}
      <div className="flex gap-2">
        {callState.status === 'idle' && (
          <button
            onClick={startCall}
            disabled={disabled}
            className="flex-1 px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 disabled:opacity-50 disabled:cursor-not-allowed text-sm font-medium"
            style={{ backgroundColor: '#2563eb', color: '#ffffff' }}
          >
            📞 Start Call
          </button>
        )}

        {callState.status === 'ringing' && (
          <>
            <button
              onClick={acceptCall}
              className="flex-1 px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 text-sm font-medium"
              style={{ backgroundColor: '#2563eb', color: '#ffffff' }}
            >
              ✅ Accept
            </button>
            <button
              onClick={() => rejectCall()}
              className="flex-1 px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 text-sm font-medium"
              style={{ backgroundColor: '#2563eb', color: '#ffffff' }}
            >
              ❌ Reject
            </button>
          </>
        )}

        {(callState.status === 'calling' || callState.status === 'connected') && (
          <>
            {callState.status === 'connected' && (
              <button
                onClick={toggleAudio}
                className={`px-4 py-2 rounded-md focus:outline-none focus:ring-2 text-sm font-medium ${
                  callState.isAudioEnabled
                    ? 'bg-blue-600 text-white hover:bg-blue-700 focus:ring-blue-500'
                    : 'bg-blue-600 text-white hover:bg-blue-700 focus:ring-blue-500'
                }`}
                style={{ backgroundColor: '#2563eb', color: '#ffffff' }}
              >
                {callState.isAudioEnabled ? '🔊 Mute' : '🔇 Unmute'}
              </button>
            )}
            <button
              onClick={endCall}
              className="flex-1 px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 text-sm font-medium"
              style={{ backgroundColor: '#2563eb', color: '#ffffff' }}
            >
              📞 End Call
            </button>
          </>
        )}

        {(callState.status === 'ended' || callState.status === 'rejected') && (
          <button
            onClick={() => window.location.reload()}
            className="flex-1 px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 text-sm font-medium"
            style={{ backgroundColor: '#2563eb', color: '#ffffff' }}
          >
            🔄 New Call
          </button>
        )}
      </div>

      {/* Disclaimer */}
      <div className="mt-3 p-2 bg-yellow-50 border border-yellow-200 rounded text-xs text-yellow-700">
        ⚠️ Audio calls may fail due to network restrictions. Ensure microphone access is allowed.
      </div>
    </div>
  );
};

export default AudioCall;