import React from 'react';
import { NegotiationEvent } from '../../types/negotiation';
import OfferCard from './OfferCard';

interface NegotiationTimelineProps {
  events: NegotiationEvent[];
  canAcceptOffer: (event: NegotiationEvent) => boolean;
  canRejectOffer: (event: NegotiationEvent) => boolean;
  onAcceptOffer: (eventId: string) => void;
  onRejectOffer: (eventId: string) => void;
  loading: boolean;
}

const NegotiationTimeline: React.FC<NegotiationTimelineProps> = ({
  events,
  canAcceptOffer,
  canRejectOffer,
  onAcceptOffer,
  onRejectOffer,
  loading
}) => {
  if (!events || events.length === 0) {
    return (
      <div className="bg-white rounded-lg shadow-sm border p-6">
        <h3 className="text-lg font-medium text-gray-900 mb-4">Timeline</h3>
        <p className="text-gray-500">No events yet.</p>
      </div>
    );
  }

  // Sort events by creation date (newest first for display)
  const sortedEvents = [...events].sort((a, b) => 
    new Date(b.created_at).getTime() - new Date(a.created_at).getTime()
  );

  return (
    <div className="bg-white rounded-lg shadow-sm border p-6">
      <h3 className="text-lg font-medium text-gray-900 mb-4">Timeline</h3>
      <div className="space-y-4">
        {sortedEvents.map((event) => (
          <OfferCard
            key={event.id}
            event={event}
            canAccept={canAcceptOffer(event)}
            canReject={canRejectOffer(event)}
            onAccept={onAcceptOffer}
            onReject={onRejectOffer}
            loading={loading}
          />
        ))}
      </div>
    </div>
  );
};

export default NegotiationTimeline;