import React from 'react';
import { NegotiationEvent } from '../../types/negotiation';

interface OfferCardProps {
  event: NegotiationEvent;
  canAccept: boolean;
  canReject: boolean;
  onAccept: (eventId: string) => void;
  onReject: (eventId: string) => void;
  loading: boolean;
}

const OfferCard: React.FC<OfferCardProps> = ({
  event,
  canAccept,
  canReject,
  onAccept,
  onReject,
  loading
}) => {
  const formatAmount = (amount: number) => {
    return new Intl.NumberFormat('en-IN', {
      style: 'currency',
      currency: 'INR',
      maximumFractionDigits: 0,
    }).format(amount);
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleString('en-IN', {
      day: 'numeric',
      month: 'short',
      year: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  const getEventTypeText = (eventType: string) => {
    switch (eventType) {
      case 'NEGOTIATION_CREATED':
        return 'Negotiation Started';
      case 'OFFER_SUBMITTED':
        return 'Offer Submitted';
      case 'COUNTER_SUBMITTED':
        return 'Counter Offer';
      case 'OFFER_ACCEPTED':
        return 'Offer Accepted';
      case 'OFFER_REJECTED':
        return 'Offer Rejected';
      case 'OFFER_EXPIRED':
        return 'Offer Expired';
      default:
        return eventType;
    }
  };

  const getRoleColor = (role: string) => {
    switch (role) {
      case 'BUYER':
        return 'text-blue-600';
      case 'SELLER':
        return 'text-green-600';
      case 'AGENT':
        return 'text-purple-600';
      case 'SYSTEM':
        return 'text-gray-600';
      default:
        return 'text-gray-600';
    }
  };

  return (
    <div className={`p-4 border rounded-lg ${event.is_active ? 'border-blue-200 bg-blue-50' : 'border-gray-200 bg-white'}`}>
      <div className="flex justify-between items-start mb-2">
        <div>
          <h4 className="font-medium text-gray-900">{getEventTypeText(event.event_type)}</h4>
          <p className={`text-sm ${getRoleColor(event.offered_by)}`}>
            by {event.offered_by}
          </p>
        </div>
        <div className="text-right">
          <p className="text-sm text-gray-500">{formatDate(event.created_at)}</p>
          {event.is_active && (
            <span className="inline-block px-2 py-1 text-xs bg-blue-100 text-blue-800 rounded-full mt-1">
              Active
            </span>
          )}
        </div>
      </div>

      {event.amount && (
        <div className="mb-2">
          <p className="text-lg font-semibold text-gray-900">
            {formatAmount(event.amount)}
          </p>
        </div>
      )}

      {event.message && (
        <div className="mb-3">
          <p className="text-sm text-gray-700 bg-gray-50 p-2 rounded">
            "{event.message}"
          </p>
        </div>
      )}

      {event.expires_at && event.is_active && (
        <div className="mb-3">
          <p className="text-xs text-orange-600">
            Expires: {formatDate(event.expires_at)}
          </p>
        </div>
      )}

      {event.is_active && event.amount && (canAccept || canReject) && (
        <div className="flex gap-2 mt-3">
          {canAccept && (
            <button
              onClick={() => onAccept(event.id)}
              disabled={loading}
              style={{ backgroundColor: '#2563eb', color: '#ffffff', padding: '8px 16px', borderRadius: '6px', fontWeight: '600' }}
              className="disabled:opacity-50 disabled:cursor-not-allowed shadow-md"
            >
              {loading ? 'Processing...' : 'Accept Offer'}
            </button>
          )}
          {canReject && (
            <button
              onClick={() => onReject(event.id)}
              disabled={loading}
              style={{ backgroundColor: '#2563eb', color: '#ffffff', padding: '8px 16px', borderRadius: '6px', fontWeight: '600' }}
              className="disabled:opacity-50 disabled:cursor-not-allowed shadow-md"
            >
              {loading ? 'Processing...' : 'Reject Offer'}
            </button>
          )}
        </div>
      )}
    </div>
  );
};

export default OfferCard;