import React, { useState } from 'react';

interface OfferComposerProps {
  onSubmitOffer: (amount: number, message?: string) => Promise<void>;
  canMakeOffer: boolean;
  loading: boolean;
}

const OfferComposer: React.FC<OfferComposerProps> = ({
  onSubmitOffer,
  canMakeOffer,
  loading
}) => {
  const [amount, setAmount] = useState<string>('');
  const [message, setMessage] = useState<string>('');
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [error, setError] = useState<string>('');

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setError('');

    // Validate amount
    const numericAmount = parseFloat(amount.replace(/,/g, ''));
    if (!numericAmount || numericAmount <= 0) {
      setError('Please enter a valid amount');
      return;
    }

    if (numericAmount < 1000) {
      setError('Amount must be at least ₹1,000');
      return;
    }

    setIsSubmitting(true);
    try {
      await onSubmitOffer(numericAmount, message.trim() || undefined);
      // Clear form on success
      setAmount('');
      setMessage('');
    } catch (error: any) {
      setError(error.message || 'Failed to submit offer');
    } finally {
      setIsSubmitting(false);
    }
  };

  const formatAmountInput = (value: string) => {
    // Remove non-numeric characters except decimal point
    const numericValue = value.replace(/[^0-9.]/g, '');
    
    // Format with commas for display
    const parts = numericValue.split('.');
    parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, ',');
    
    return parts.join('.');
  };

  const handleAmountChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const formatted = formatAmountInput(e.target.value);
    setAmount(formatted);
  };

  if (!canMakeOffer) {
    return (
      <div className="bg-gray-50 rounded-lg shadow-sm border p-6">
        <h3 className="text-lg font-medium text-gray-900 mb-2">Make Offer</h3>
        <p className="text-gray-500">
          This negotiation has ended. No new offers can be submitted.
        </p>
      </div>
    );
  }

  return (
    <div className="bg-white rounded-lg shadow-sm border p-6">
      <h3 className="text-lg font-medium text-gray-900 mb-4">Make an Offer</h3>
      
      <form onSubmit={handleSubmit} className="space-y-4">
        <div>
          <label htmlFor="amount" className="block text-sm font-medium text-gray-700 mb-2">
            Offer Amount (₹)
          </label>
          <input
            type="text"
            id="amount"
            value={amount}
            onChange={handleAmountChange}
            placeholder="Enter your offer amount"
            className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            required
            disabled={loading || isSubmitting}
          />
          <p className="text-xs text-gray-500 mt-1">
            Enter amount in Indian Rupees (e.g., 50,00,000)
          </p>
        </div>

        <div>
          <label htmlFor="message" className="block text-sm font-medium text-gray-700 mb-2">
            Message (Optional)
          </label>
          <textarea
            id="message"
            value={message}
            onChange={(e) => setMessage(e.target.value)}
            placeholder="Add a personal message with your offer..."
            rows={3}
            className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent resize-none"
            disabled={loading || isSubmitting}
            maxLength={500}
          />
          <p className="text-xs text-gray-500 mt-1">
            {message.length}/500 characters
          </p>
        </div>

        {error && (
          <div className="p-3 bg-red-50 border border-red-200 rounded-md">
            <p className="text-sm text-red-600">{error}</p>
          </div>
        )}

        <button
          type="submit"
          disabled={loading || isSubmitting || !amount.trim()}
          style={{ 
            backgroundColor: '#2563eb', 
            color: '#ffffff', 
            padding: '12px 24px', 
            borderRadius: '6px', 
            fontWeight: '600',
            width: '100%',
            cursor: loading || isSubmitting || !amount.trim() ? 'not-allowed' : 'pointer',
            opacity: loading || isSubmitting || !amount.trim() ? 0.5 : 1
          }}
        >
          {isSubmitting ? 'Submitting Offer...' : 'Submit Offer'}
        </button>
      </form>
    </div>
  );
};

export default OfferComposer;