import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '../ui/card';
import { Button } from '../ui/button';
import { Input } from '../ui/input';
import { Label } from '../ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '../ui/select';
import { Upload } from 'lucide-react';
import { ownerKYCApi } from '../../api/ownerKYC';
import { toast } from 'sonner';

const DOCUMENT_TYPES = [
  { value: 'pan_document', label: 'PAN Card' },
  { value: 'aadhaar_document', label: 'Aadhaar Card' },
  { value: 'photo', label: 'Photo' },
  { value: 'sale_deed', label: 'Sale Deed' },
  { value: 'property_tax_receipt', label: 'Property Tax Receipt' },
  { value: 'encumbrance_certificate', label: 'Encumbrance Certificate' },
  { value: 'other_ownership_proof', label: 'Other Ownership Proof' },
];

interface KYCDocumentUploadProps {
  onUploadComplete?: () => void;
  userId?: string;
}

export const KYCDocumentUpload: React.FC<KYCDocumentUploadProps> = ({ onUploadComplete, userId }) => {
  const [selectedFile, setSelectedFile] = useState<File | null>(null);
  const [documentType, setDocumentType] = useState<string>('');
  const [uploading, setUploading] = useState(false);

  const handleFileSelect = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      const allowedTypes = ['image/jpeg', 'image/png', 'image/jpg', 'application/pdf'];
      if (!allowedTypes.includes(file.type)) {
        toast.error('Only JPG, PNG, and PDF files are allowed');
        return;
      }
      
      if (file.size > 5 * 1024 * 1024) {
        toast.error('File size must be less than 5MB');
        return;
      }
      
      setSelectedFile(file);
    }
  };

  const handleUpload = async () => {
    if (!selectedFile || !documentType) {
      toast.error('Please select a file and document type');
      return;
    }

    setUploading(true);
    try {
      await ownerKYCApi.uploadKYCDocument(selectedFile, documentType, userId);
      toast.success('Document uploaded successfully');
      setSelectedFile(null);
      setDocumentType('');
      onUploadComplete?.();
    } catch (error: any) {
      toast.error(error.message || 'Upload failed');
    } finally {
      setUploading(false);
    }
  };

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <Upload className="h-5 w-5" />
          KYC Document Upload
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-4">
        <div>
          <Label htmlFor="document-type">Document Type *</Label>
          <Select value={documentType} onValueChange={setDocumentType}>
            <SelectTrigger>
              <SelectValue placeholder="Select document type" />
            </SelectTrigger>
            <SelectContent>
              {DOCUMENT_TYPES.map((type) => (
                <SelectItem key={type.value} value={type.value}>
                  {type.label}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        <div>
          <Label htmlFor="file-upload">Upload Document *</Label>
          <Input
            id="file-upload"
            type="file"
            accept=".jpg,.jpeg,.png,.pdf"
            onChange={handleFileSelect}
            disabled={uploading}
          />
          {selectedFile && (
            <p className="text-sm text-gray-600 mt-1">
              Selected: {selectedFile.name} ({(selectedFile.size / 1024 / 1024).toFixed(2)} MB)
            </p>
          )}
        </div>

        <Button 
          onClick={handleUpload} 
          disabled={!selectedFile || !documentType || uploading}
          className="w-full"
        >
          {uploading ? 'Uploading...' : 'Upload Document'}
        </Button>

        <div className="text-xs text-gray-500 space-y-1">
          <p>• Supported formats: JPG, PNG, PDF</p>
          <p>• Maximum file size: 5MB</p>
          <p>• Ensure documents are clear and readable</p>
        </div>
      </CardContent>
    </Card>
  );
};