import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '../ui/card';
import { Button } from '../ui/button';
import { Input } from '../ui/input';
import { Label } from '../ui/label';
import { Shield, CheckCircle } from 'lucide-react';
import { toast } from 'sonner';

export const KYCProfileForm: React.FC = () => {
  const [aadhaarNumber, setAadhaarNumber] = useState('');
  const [panNumber, setPanNumber] = useState('');
  const [fullName, setFullName] = useState('');
  const [dateOfBirth, setDateOfBirth] = useState('');
  const [aadhaarOtp, setAadhaarOtp] = useState('');
  const [panOtp, setPanOtp] = useState('');
  const [aadhaarOtpSent, setAadhaarOtpSent] = useState(false);
  const [panOtpSent, setPanOtpSent] = useState(false);
  const [aadhaarVerified, setAadhaarVerified] = useState(false);
  const [panVerified, setPanVerified] = useState(false);
  const [loading, setLoading] = useState(false);
  const [kycCompleted, setKycCompleted] = useState(false);
  const [existingKycData, setExistingKycData] = useState(null);

  useEffect(() => {
    fetchKycProfile();
  }, []);

  const fetchKycProfile = async () => {
    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const userId = user.user_id || 'b5589fd0-4935-4b83-a514-59c5be575ece';
      
      const response = await fetch(`http://localhost:8090/api/user-account/kyc/profile?user_id=${userId}`);
      const data = await response.json();
      
      if (data.success && data.kyc_data) {
        setExistingKycData(data.kyc_data);
        setKycCompleted(data.kyc_data.verification_status === 'verified');
        setFullName(data.kyc_data.full_name);
        setDateOfBirth(data.kyc_data.date_of_birth);
        setAadhaarNumber(data.kyc_data.aadhaar_number);
        setPanNumber(data.kyc_data.pan_number);
        setAadhaarVerified(data.kyc_data.aadhaar_verified);
        setPanVerified(data.kyc_data.pan_verified);
      }
    } catch (error) {
      console.error('Error fetching KYC profile:', error);
    }
  };

  const sendAadhaarOtp = async () => {
    if (aadhaarNumber.length !== 12) {
      toast.error('Please enter a valid 12-digit Aadhaar number');
      return;
    }
    
    setLoading(true);
    try {
      const response = await fetch('http://localhost:8090/api/user-account/kyc/verify/aadhaar/send-otp', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ aadhaar_number: aadhaarNumber })
      });
      
      const data = await response.json();
      
      if (response.ok) {
        setAadhaarOtpSent(true);
        toast.success('OTP sent to your registered mobile number');
        if (data.demo_otp) {
          console.log('Demo Aadhaar OTP:', data.demo_otp);
        }
      } else {
        toast.error(data.detail || 'Failed to send OTP. Please try again.');
      }
    } catch (error) {
      console.error('Aadhaar OTP Error:', error);
      toast.error('Network error. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  const verifyAadhaarOtp = async () => {
    if (aadhaarOtp.length !== 6) {
      toast.error('Please enter a valid 6-digit OTP');
      return;
    }
    
    setLoading(true);
    try {
      const response = await fetch('http://localhost:8090/api/user-account/kyc/verify/aadhaar/verify-otp', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ 
          aadhaar_number: aadhaarNumber,
          otp: aadhaarOtp 
        })
      });
      
      if (response.ok) {
        setAadhaarVerified(true);
        toast.success('Aadhaar verified successfully!');
      } else {
        toast.error('Invalid OTP. Please try again.');
      }
    } catch (error) {
      toast.error('Network error. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  const sendPanOtp = async () => {
    if (!panNumber || !fullName || !dateOfBirth) {
      toast.error('Please fill all PAN verification fields');
      return;
    }
    
    setLoading(true);
    try {
      const response = await fetch('http://localhost:8090/api/user-account/kyc/verify/pan/send-otp', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          pan_number: panNumber,
          name: fullName,
          date_of_birth: dateOfBirth
        })
      });
      
      const data = await response.json();
      
      if (response.ok) {
        setPanOtpSent(true);
        toast.success('OTP sent for PAN verification');
        if (data.demo_otp) {
          console.log('Demo PAN OTP:', data.demo_otp);
        }
      } else {
        toast.error(data.detail || 'Failed to send OTP. Please check your details.');
      }
    } catch (error) {
      console.error('PAN OTP Error:', error);
      toast.error('Network error. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  const verifyPanOtp = async () => {
    if (panOtp.length !== 6) {
      toast.error('Please enter a valid 6-digit OTP');
      return;
    }
    
    setLoading(true);
    try {
      const response = await fetch('http://localhost:8090/api/user-account/kyc/verify/pan/verify-otp', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ 
          pan_number: panNumber,
          otp: panOtp 
        })
      });
      
      if (response.ok) {
        setPanVerified(true);
        toast.success('PAN verified successfully!');
      } else {
        toast.error('Invalid OTP. Please try again.');
      }
    } catch (error) {
      toast.error('Network error. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  const handleSave = async () => {
    if (!aadhaarVerified || !panVerified) {
      toast.error('Please verify both Aadhaar and PAN before saving');
      return;
    }
    
    const user = JSON.parse(localStorage.getItem('user') || '{}');
    const userId = user.user_id;
    
    if (!userId) {
      toast.error('User not found. Please login again.');
      return;
    }
    
    setLoading(true);
    try {
      const response = await fetch('http://localhost:8090/api/user-account/kyc/save', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          user_id: userId,
          full_name: fullName,
          date_of_birth: dateOfBirth,
          aadhaar_number: aadhaarNumber,
          pan_number: panNumber,
          aadhaar_verified: aadhaarVerified,
          pan_verified: panVerified
        })
      });
      
      if (response.ok) {
        toast.success('Profile and verification data saved successfully!');
        setKycCompleted(true);
        fetchKycProfile();
        // Notify profile page to refresh KYC status
        window.dispatchEvent(new Event('kycUpdated'));
        // Scroll to top after successful submission
        window.scrollTo({ top: 0, behavior: 'smooth' });
      } else {
        toast.error('Failed to save profile data. Please try again.');
      }
    } catch (error) {
      toast.error('Network error. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  if (kycCompleted && existingKycData && existingKycData.verification_status === 'verified') {
    return (
      <div className="space-y-6">
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <CheckCircle className="h-5 w-5 text-green-500" />
              KYC Verification Completed
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="bg-green-50 p-4 rounded-lg">
              <p className="text-green-800 font-medium">Your KYC verification is complete!</p>
              <p className="text-green-600 text-sm mt-1">All your documents have been verified successfully.</p>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <Label>Full Name</Label>
                <p className="font-medium">{existingKycData.full_name}</p>
              </div>
              <div>
                <Label>Date of Birth</Label>
                <p className="font-medium">{existingKycData.date_of_birth}</p>
              </div>
              <div>
                <Label>Aadhaar Number</Label>
                <p className="font-medium">****-****-{existingKycData.aadhaar_number.slice(-4)}</p>
              </div>
              <div>
                <Label>PAN Number</Label>
                <p className="font-medium">{existingKycData.pan_number}</p>
              </div>
            </div>
            <Button 
              type="button"
              onClick={() => {
                setKycCompleted(false);
                setExistingKycData(null);
              }}
              variant="outline"
              className="w-full"
            >
              Update KYC Information
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Shield className="h-5 w-5 text-blue-600" />
            Profile & Document Information
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-6">
          {/* Personal Information */}
          <div className="space-y-4">
            <h3 className="text-lg font-semibold">Personal Information</h3>
            <div className="flex gap-4">
              <div className="flex-1 space-y-3">
                <Label htmlFor="fullName">Full Name (as per documents)</Label>
                <Input
                  id="fullName"
                  value={fullName}
                  onChange={(e) => setFullName(e.target.value)}
                  placeholder="Enter your full name"
                  className="w-full h-12 text-lg px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  style={{ width: '100%' }}
                />
              </div>
              <div className="flex-1 space-y-3">
                <Label htmlFor="dob">Date of Birth</Label>
                <Input
                  id="dob"
                  type="date"
                  value={dateOfBirth}
                  onChange={(e) => setDateOfBirth(e.target.value)}
                  className="w-full h-12 text-lg px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  style={{ width: '100%' }}
                />
              </div>
            </div>
          </div>

          {/* Document Information */}
          <div className="space-y-4">
            <h3 className="text-lg font-semibold">Document Information</h3>
            <div className="space-y-4">
              <div className="space-y-3">
                <div className="flex items-center gap-2">
                  <Label htmlFor="aadhaar">Aadhaar Number</Label>
                  {aadhaarVerified && <CheckCircle className="h-4 w-4 text-green-500" />}
                </div>
                <Input
                  id="aadhaar"
                  value={aadhaarNumber}
                  onChange={(e) => setAadhaarNumber(e.target.value.replace(/\D/g, '').slice(0, 12))}
                  placeholder="Enter 12-digit Aadhaar number"
                  maxLength={12}
                  disabled={aadhaarVerified}
                  className="h-12 text-lg"
                />
                {!aadhaarOtpSent ? (
                  <Button 
                    type="button"
                    onClick={sendAadhaarOtp} 
                    disabled={loading || aadhaarNumber.length !== 12 || aadhaarVerified}
                    size="sm"
                    className="bg-[#0056D2] hover:bg-[#0056D2]/90 text-white"
                  >
                    {loading ? 'Sending...' : 'Send OTP'}
                  </Button>
                ) : !aadhaarVerified ? (
                  <div className="space-y-2">
                    <Input
                      value={aadhaarOtp}
                      onChange={(e) => setAadhaarOtp(e.target.value.replace(/\D/g, '').slice(0, 6))}
                      placeholder="Enter 6-digit OTP"
                      maxLength={6}
                    />
                    <Button 
                      type="button"
                      onClick={verifyAadhaarOtp} 
                      disabled={loading || aadhaarOtp.length !== 6}
                      size="sm"
                      className="bg-[#0056D2] hover:bg-[#0056D2]/90 text-white"
                    >
                      {loading ? 'Verifying...' : 'Verify OTP'}
                    </Button>
                  </div>
                ) : null}
                <p className="text-xs text-gray-500">
                  {aadhaarVerified ? 'Aadhaar verified successfully' : 'Enter your 12-digit Aadhaar number for identity verification'}
                </p>
              </div>
              
              <div className="space-y-3">
                <div className="flex items-center gap-2">
                  <Label htmlFor="pan">PAN Number</Label>
                  {panVerified && <CheckCircle className="h-4 w-4 text-green-500" />}
                </div>
                <Input
                  id="pan"
                  value={panNumber}
                  onChange={(e) => setPanNumber(e.target.value.toUpperCase().slice(0, 10))}
                  placeholder="Enter PAN number (e.g., ABCDE1234F)"
                  maxLength={10}
                  disabled={panVerified}
                  className="h-12 text-lg"
                />
                {!panOtpSent ? (
                  <Button 
                    type="button"
                    onClick={sendPanOtp} 
                    disabled={loading || !panNumber || !fullName || !dateOfBirth || panVerified}
                    size="sm"
                    className="bg-[#0056D2] hover:bg-[#0056D2]/90 text-white"
                  >
                    {loading ? 'Sending...' : 'Send OTP'}
                  </Button>
                ) : !panVerified ? (
                  <div className="space-y-2">
                    <Input
                      value={panOtp}
                      onChange={(e) => setPanOtp(e.target.value.replace(/\D/g, '').slice(0, 6))}
                      placeholder="Enter 6-digit OTP"
                      maxLength={6}
                    />
                    <Button 
                      type="button"
                      onClick={verifyPanOtp} 
                      disabled={loading || panOtp.length !== 6}
                      size="sm"
                      className="bg-[#0056D2] hover:bg-[#0056D2]/90 text-white"
                    >
                      {loading ? 'Verifying...' : 'Verify OTP'}
                    </Button>
                  </div>
                ) : null}
                <p className="text-xs text-gray-500">
                  {panVerified ? 'PAN verified successfully' : 'Enter your 10-character PAN number'}
                </p>
              </div>
            </div>
          </div>

          <Button 
            type="button"
            onClick={handleSave} 
            className="w-full bg-[#0056D2] hover:bg-[#0056D2]/90 text-white" 
            disabled={!aadhaarVerified || !panVerified || loading}
          >
            {loading ? 'Saving...' : (existingKycData ? 'Update Profile & Document Information' : 'Save Profile & Document Information')}
          </Button>
          
          {(!aadhaarVerified || !panVerified) && (
            <p className="text-sm text-gray-500 text-center">
              Please verify both Aadhaar and PAN to save your information
            </p>
          )}
        </CardContent>
      </Card>
    </div>
  );
};