import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '../ui/card';
import { Badge } from '../ui/badge';
import { Progress } from '../ui/progress';
import { CheckCircle, XCircle, Clock, FileText } from 'lucide-react';
import { ownerKYCApi, KYCStatusResponse } from '../../api/ownerKYC';
import { toast } from 'sonner';

export const KYCStatus: React.FC = () => {
  const [kycStatus, setKycStatus] = useState<KYCStatusResponse | null>(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchKYCStatus();
  }, []);

  const fetchKYCStatus = async () => {
    try {
      const status = await ownerKYCApi.getKYCStatus();
      setKycStatus(status);
    } catch (error: any) {
      if (error.message.includes('404')) {
        // No KYC profile found
        setKycStatus(null);
      } else {
        toast.error('Failed to fetch KYC status');
      }
    } finally {
      setLoading(false);
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'verified':
        return <CheckCircle className="h-4 w-4 text-green-500" />;
      case 'rejected':
        return <XCircle className="h-4 w-4 text-red-500" />;
      default:
        return <Clock className="h-4 w-4 text-yellow-500" />;
    }
  };

  const getStatusBadge = (status: string) => {
    const variants = {
      verified: 'default',
      rejected: 'destructive',
      pending: 'secondary',
      in_progress: 'secondary',
    } as const;
    
    return (
      <Badge variant={variants[status as keyof typeof variants] || 'secondary'}>
        {status.replace('_', ' ').toUpperCase()}
      </Badge>
    );
  };

  if (loading) {
    return (
      <Card>
        <CardContent className="p-8 text-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-2 text-gray-600">Loading KYC status...</p>
        </CardContent>
      </Card>
    );
  }

  if (!kycStatus) {
    return (
      <Card>
        <CardContent className="p-8 text-center">
          <FileText className="h-12 w-12 text-gray-300 mx-auto mb-4" />
          <h3 className="text-lg font-semibold text-gray-900 mb-2">No KYC Profile Found</h3>
          <p className="text-gray-600">Please create your KYC profile first to view status.</p>
        </CardContent>
      </Card>
    );
  }

  return (
    <div className="space-y-6">
      {/* Overall Status */}
      <Card>
        <CardHeader>
          <CardTitle>KYC Verification Status</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            <div className="flex items-center justify-between">
              <span className="text-sm font-medium">Overall Progress</span>
              <span className="text-sm text-gray-600">{kycStatus.overall_progress}%</span>
            </div>
            <Progress value={kycStatus.overall_progress} className="w-full" />
            
            <div className="flex items-center gap-2">
              <span className="text-sm font-medium">Status:</span>
              {getStatusBadge(kycStatus.kyc_profile.kyc_status)}
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Profile Information */}
      <Card>
        <CardHeader>
          <CardTitle>Profile Information</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <p className="text-sm font-medium text-gray-600">Full Name</p>
              <p className="text-sm">{kycStatus.kyc_profile.full_name}</p>
            </div>
            <div>
              <p className="text-sm font-medium text-gray-600">Email</p>
              <p className="text-sm">{kycStatus.kyc_profile.email}</p>
            </div>
            <div>
              <p className="text-sm font-medium text-gray-600">Phone</p>
              <p className="text-sm">{kycStatus.kyc_profile.phone_number}</p>
            </div>
            <div>
              <p className="text-sm font-medium text-gray-600">Aadhaar Number</p>
              <p className="text-sm">{kycStatus.kyc_profile.aadhaar_number ? `****-****-${kycStatus.kyc_profile.aadhaar_number.slice(-4)}` : 'Not provided'}</p>
            </div>
            <div>
              <p className="text-sm font-medium text-gray-600">PAN Number</p>
              <p className="text-sm">{kycStatus.kyc_profile.pan_number ? `${kycStatus.kyc_profile.pan_number.slice(0, 3)}****${kycStatus.kyc_profile.pan_number.slice(-1)}` : 'Not provided'}</p>
            </div>
            <div>
              <p className="text-sm font-medium text-gray-600">Submitted On</p>
              <p className="text-sm">{new Date(kycStatus.kyc_profile.created_at).toLocaleDateString()}</p>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Next Steps */}
      <Card>
        <CardHeader>
          <CardTitle>Next Steps</CardTitle>
        </CardHeader>
        <CardContent>
          <ul className="space-y-2">
            {kycStatus.next_steps.map((step, index) => (
              <li key={index} className="flex items-center gap-2">
                <div className="w-2 h-2 bg-blue-500 rounded-full"></div>
                <span className="text-sm">{step}</span>
              </li>
            ))}
          </ul>
        </CardContent>
      </Card>
    </div>
  );
};