import React, { useState } from 'react';
import { CheckCircle, ArrowRight, ArrowLeft, AlertCircle } from 'lucide-react';
import { Button } from '../ui/button';
import { Input } from '../ui/input';
import { Label } from '../ui/label';
import { Card } from '../ui/card';
import { toast } from 'sonner';
import { ownerKYCApi } from '../../api/ownerKYC';
import axios from 'axios';

const API_BASE_URL = import.meta.env.VITE_API_BASE_URL || 'http://localhost:8090';

export const OwnerKYCForm: React.FC = () => {
  const [currentStep, setCurrentStep] = useState(1);
  const [loading, setLoading] = useState(false);
  const [formData, setFormData] = useState({
    user_id: '',
    full_name: '',
    pan_number: '',
    aadhaar_number: '',
    date_of_birth: '',
    phone_number: '',
    email: '',
    address: '',
    city: '',
    state: '',
    pincode: '',
    pan_document_url: '',
    aadhaar_document_url: '',
    photo_url: '',
    sale_deed_url: '',
    property_tax_receipt_url: '',
    encumbrance_certificate_url: '',
    other_ownership_proof_url: ''
  });

  // Generate user_id from email when email changes
  React.useEffect(() => {
    if (formData.email) {
      // Simple hash function to generate consistent user_id from email
      const userId = btoa(formData.email).replace(/[^a-zA-Z0-9]/g, '').substring(0, 16);
      setFormData(prev => ({ ...prev, user_id: userId }));
    }
  }, [formData.email]);

  const getToken = () => localStorage.getItem('token') || localStorage.getItem('access_token');

  const uploadFile = async (file: File, docType: string) => {
    try {
      const response = await ownerKYCApi.uploadKYCDocument(file, docType, formData.user_id);
      return response.success ? 'uploaded' : null;
    } catch (error) {
      throw error;
    }
  };

  const handleFileChange = async (type: string, e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;

    if (!formData.user_id) {
      toast.error('Please enter your email first');
      return;
    }

    setLoading(true);
    try {
      const result = await uploadFile(file, type);
      if (result) {
        setFormData(prev => ({ ...prev, [`${type}_url`]: 'uploaded' }));
        toast.success(`${type.replace('_', ' ')} uploaded successfully`);
      }
    } catch (error: any) {
      toast.error(error.message || 'Upload failed');
    } finally {
      setLoading(false);
    }
  };

  const submitKYC = async () => {
    if (!formData.pan_document_url) {
      toast.error('PAN document is mandatory');
      return;
    }

    setLoading(true);
    try {
      await ownerKYCApi.submitKYC(formData);
      toast.success('Owner KYC submitted successfully!');
      setTimeout(() => window.location.href = '/list-property', 2000);
    } catch (error: any) {
      toast.error(error.message || 'Submission failed');
    } finally {
      setLoading(false);
    }
  };

  const steps = [
    { number: 1, title: 'Identity' },
    { number: 2, title: 'Address' },
    { number: 3, title: 'Documents' },
    { number: 4, title: 'Ownership' }
  ];

  return (
    <div className="max-w-4xl mx-auto p-6">
      <h1 className="text-3xl font-bold mb-2">Owner KYC Verification</h1>
      <p className="text-gray-600 mb-8">Required to list properties</p>

      {/* Progress */}
      <div className="flex justify-between mb-12">
        {steps.map((step, idx) => (
          <div key={step.number} className="flex items-center flex-1">
            <div className="flex flex-col items-center flex-1">
              <div className={`w-10 h-10 rounded-full flex items-center justify-center font-semibold
                ${currentStep > step.number ? 'bg-green-500 text-white' : 
                  currentStep === step.number ? 'bg-blue-500 text-white ring-4 ring-blue-200' : 
                  'bg-gray-200 text-gray-600'}`}>
                {currentStep > step.number ? <CheckCircle className="w-6 h-6" /> : step.number}
              </div>
              <span className="text-sm mt-2 font-medium">{step.title}</span>
            </div>
            {idx < steps.length - 1 && (
              <div className={`h-1 flex-1 mx-2 ${currentStep > step.number ? 'bg-green-500' : 'bg-gray-200'}`} />
            )}
          </div>
        ))}
      </div>

      <Card className="p-6">
        {/* Step 1: Identity */}
        {currentStep === 1 && (
          <div className="space-y-4">
            <h2 className="text-xl font-semibold mb-4">Identity Information</h2>
            <div><Label>Full Name *</Label><Input value={formData.full_name} onChange={(e) => setFormData(prev => ({ ...prev, full_name: e.target.value }))} /></div>
            <div><Label>PAN *</Label><Input value={formData.pan_number} onChange={(e) => setFormData(prev => ({ ...prev, pan_number: e.target.value.toUpperCase() }))} maxLength={10} /></div>
            <div><Label>Aadhaar</Label><Input value={formData.aadhaar_number} onChange={(e) => setFormData(prev => ({ ...prev, aadhaar_number: e.target.value }))} maxLength={12} /></div>
            <div><Label>DOB *</Label><Input type="date" value={formData.date_of_birth} onChange={(e) => setFormData(prev => ({ ...prev, date_of_birth: e.target.value }))} /></div>
            <div><Label>Phone *</Label><Input value={formData.phone_number} onChange={(e) => setFormData(prev => ({ ...prev, phone_number: e.target.value }))} maxLength={10} /></div>
            <div><Label>Email *</Label><Input type="email" value={formData.email} onChange={(e) => setFormData(prev => ({ ...prev, email: e.target.value }))} /></div>
          </div>
        )}

        {/* Step 2: Address */}
        {currentStep === 2 && (
          <div className="space-y-4">
            <h2 className="text-xl font-semibold mb-4">Address</h2>
            <div><Label>Address *</Label><Input value={formData.address} onChange={(e) => setFormData(prev => ({ ...prev, address: e.target.value }))} /></div>
            <div className="grid grid-cols-2 gap-4">
              <div><Label>City *</Label><Input value={formData.city} onChange={(e) => setFormData(prev => ({ ...prev, city: e.target.value }))} /></div>
              <div><Label>State *</Label><Input value={formData.state} onChange={(e) => setFormData(prev => ({ ...prev, state: e.target.value }))} /></div>
            </div>
            <div><Label>Pincode *</Label><Input value={formData.pincode} onChange={(e) => setFormData(prev => ({ ...prev, pincode: e.target.value }))} maxLength={6} /></div>
          </div>
        )}

        {/* Step 3: Identity Documents */}
        {currentStep === 3 && (
          <div className="space-y-4">
            <h2 className="text-xl font-semibold mb-4">Identity Documents</h2>
            <div className="border rounded-lg p-4">
              <Label>PAN Card * (Required)</Label>
              <div className="flex items-center gap-2 mt-2">
                {formData.pan_document_url && <CheckCircle className="w-5 h-5 text-green-500" />}
                <Input type="file" accept=".pdf,.jpg,.jpeg,.png" onChange={(e) => handleFileChange('pan_document', e)} />
              </div>
            </div>
            <div className="border rounded-lg p-4">
              <Label>Aadhaar Card</Label>
              <div className="flex items-center gap-2 mt-2">
                {formData.aadhaar_document_url && <CheckCircle className="w-5 h-5 text-green-500" />}
                <Input type="file" accept=".pdf,.jpg,.jpeg,.png" onChange={(e) => handleFileChange('aadhaar_document', e)} />
              </div>
            </div>
            <div className="border rounded-lg p-4">
              <Label>Photo</Label>
              <div className="flex items-center gap-2 mt-2">
                {formData.photo_url && <CheckCircle className="w-5 h-5 text-green-500" />}
                <Input type="file" accept=".jpg,.jpeg,.png" onChange={(e) => handleFileChange('photo', e)} />
              </div>
            </div>
          </div>
        )}

        {/* Step 4: Ownership Proof */}
        {currentStep === 4 && (
          <div className="space-y-4">
            <h2 className="text-xl font-semibold mb-4">Property Ownership Proof</h2>
            <div className="border rounded-lg p-4">
              <Label>Sale Deed</Label>
              <div className="flex items-center gap-2 mt-2">
                {formData.sale_deed_url && <CheckCircle className="w-5 h-5 text-green-500" />}
                <Input type="file" accept=".pdf" onChange={(e) => handleFileChange('sale_deed', e)} />
              </div>
            </div>
            <div className="border rounded-lg p-4">
              <Label>Property Tax Receipt</Label>
              <div className="flex items-center gap-2 mt-2">
                {formData.property_tax_receipt_url && <CheckCircle className="w-5 h-5 text-green-500" />}
                <Input type="file" accept=".pdf" onChange={(e) => handleFileChange('property_tax_receipt', e)} />
              </div>
            </div>
            <div className="border rounded-lg p-4">
              <Label>Encumbrance Certificate</Label>
              <div className="flex items-center gap-2 mt-2">
                {formData.encumbrance_certificate_url && <CheckCircle className="w-5 h-5 text-green-500" />}
                <Input type="file" accept=".pdf" onChange={(e) => handleFileChange('encumbrance_certificate', e)} />
              </div>
            </div>
            {!formData.pan_document_url && (
              <div className="flex items-center gap-2 text-amber-600 bg-amber-50 p-3 rounded">
                <AlertCircle className="w-5 h-5" />
                <span className="text-sm">PAN document is mandatory</span>
              </div>
            )}
          </div>
        )}

        {/* Navigation */}
        <div className="flex justify-between mt-8">
          {currentStep > 1 && (
            <Button variant="outline" onClick={() => setCurrentStep(currentStep - 1)} disabled={loading} className="border-[#0056D2] text-[#0056D2] hover:bg-[#0056D2] hover:text-white">
              <ArrowLeft className="w-4 h-4 mr-2" />
              Previous
            </Button>
          )}
          <div className="ml-auto">
            {currentStep < 4 ? (
              <Button onClick={() => setCurrentStep(currentStep + 1)} disabled={loading} className="bg-[#0056D2] hover:bg-[#0056D2]/90 text-white">
                Next
                <ArrowRight className="w-4 h-4 ml-2" />
              </Button>
            ) : (
              <Button onClick={submitKYC} disabled={loading || !formData.pan_document_url} className="bg-[#0056D2] hover:bg-[#0056D2]/90 text-white">
                {loading ? 'Submitting...' : 'Submit for Review'}
              </Button>
            )}
          </div>
        </div>
      </Card>
    </div>
  );
};
