import React, { useState, useEffect } from 'react';
import { Button } from '../ui/button';
import { ArrowRight, Users, Building, Phone, Mail } from 'lucide-react';
import { toast } from 'react-hot-toast';

interface Lead {
  id: number;
  buyer_name: string;
  buyer_email: string;
  buyer_phone?: string;
  property_id: number;
  source: string;
  priority: string;
  created_at: string;
  property_title?: string;
}

interface LeadToPipelineConverterProps {
  onConvert: () => void;
}

const LeadToPipelineConverter: React.FC<LeadToPipelineConverterProps> = ({ onConvert }) => {
  const [leads, setLeads] = useState<Lead[]>([]);
  const [loading, setLoading] = useState(false);
  const [converting, setConverting] = useState(false);

  useEffect(() => {
    loadLeads();
  }, []);

  const loadLeads = async () => {
    setLoading(true);
    try {
      const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
      const response = await fetch(`${API_BASE}/leads`, {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('access_token') || 'dummy-token'}`
        }
      });
      const data = await response.json();
      setLeads(data.leads || []);
    } catch (error) {
      console.error('Failed to load leads:', error);
    } finally {
      setLoading(false);
    }
  };

  const convertLeadsToPipeline = async () => {
    if (leads.length === 0) {
      toast.error('No leads to convert');
      return;
    }

    setConverting(true);
    try {
      // Convert each lead to pipeline format
      for (const lead of leads) {
        const pipelineData = {
          property_id: lead.property_id,
          buyer_name: lead.buyer_name,
          buyer_email: lead.buyer_email,
          buyer_phone: lead.buyer_phone,
          source: lead.source,
          current_stage: 'New',
          priority: lead.priority,
          ai_score: Math.floor(Math.random() * 100), // Generate AI score
          budget_min: 1000000, // Default budget
          budget_max: 5000000,
          city: 'Mumbai' // Default city
        };

        // Create pipeline entry
        const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
        await fetch(`${API_BASE}/pipeline/create-from-lead`, {
          method: 'POST',
          headers: { 
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${localStorage.getItem('access_token') || 'dummy-token'}`
          },
          body: JSON.stringify(pipelineData)
        });
      }

      toast.success(`${leads.length} leads converted to pipeline!`);
      onConvert();
    } catch (error) {
      toast.error('Failed to convert leads');
      console.error(error);
    } finally {
      setConverting(false);
    }
  };

  if (loading) {
    return (
      <div className="text-center py-8">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto"></div>
        <p className="mt-2 text-gray-600">Loading leads...</p>
      </div>
    );
  }

  return (
    <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
      <div className="flex items-center gap-3 mb-4">
        <Users className="text-blue-600" size={24} />
        <div>
          <h3 className="text-lg font-semibold text-gray-900">Convert Leads to Pipeline</h3>
          <p className="text-sm text-gray-600">Move your existing leads into the sales pipeline</p>
        </div>
      </div>

      {leads.length === 0 ? (
        <div className="text-center py-8 text-gray-500">
          <Building size={48} className="mx-auto mb-4 text-gray-300" />
          <p>No leads found to convert</p>
          <p className="text-sm mt-1">Create some leads first from property inquiries</p>
        </div>
      ) : (
        <div>
          <div className="mb-4">
            <p className="text-sm text-gray-600 mb-3">
              Found <strong>{leads.length}</strong> leads ready to convert:
            </p>
            <div className="max-h-40 overflow-y-auto space-y-2">
              {leads.slice(0, 5).map((lead) => (
                <div key={lead.id} className="flex items-center gap-3 p-2 bg-gray-50 rounded">
                  <div className="flex-1">
                    <p className="font-medium text-sm">{lead.buyer_name}</p>
                    <div className="flex items-center gap-4 text-xs text-gray-600">
                      <span className="flex items-center gap-1">
                        <Mail size={12} />
                        {lead.buyer_email}
                      </span>
                      {lead.buyer_phone && (
                        <span className="flex items-center gap-1">
                          <Phone size={12} />
                          {lead.buyer_phone}
                        </span>
                      )}
                    </div>
                  </div>
                  <span className="text-xs bg-blue-100 text-blue-700 px-2 py-1 rounded">
                    {lead.source}
                  </span>
                </div>
              ))}
              {leads.length > 5 && (
                <p className="text-xs text-gray-500 text-center">
                  +{leads.length - 5} more leads...
                </p>
              )}
            </div>
          </div>

          <Button
            onClick={convertLeadsToPipeline}
            disabled={converting}
            className="w-full bg-blue-600 hover:bg-blue-700 text-white"
          >
            {converting ? (
              <div className="flex items-center gap-2">
                <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white"></div>
                Converting...
              </div>
            ) : (
              <div className="flex items-center gap-2">
                <ArrowRight size={16} />
                Convert {leads.length} Leads to Pipeline
              </div>
            )}
          </Button>
        </div>
      )}
    </div>
  );
};

export default LeadToPipelineConverter;