import React, { useState } from 'react';
import { LeadPipeline } from '../../types/pipeline';
import { Clock, User, DollarSign, MapPin, TrendingUp, AlertCircle, Phone, Mail, Calendar, MoreVertical } from 'lucide-react';
import { formatDistanceToNow, format } from 'date-fns';
import ActivityTimeline from './ActivityTimeline';

interface PipelineCardProps {
  lead: LeadPipeline;
  isDragging: boolean;
  onRefresh: () => void;
}

const PipelineCard: React.FC<PipelineCardProps> = ({ lead, isDragging, onRefresh }) => {
  const [showDetails, setShowDetails] = useState(false);
  const [showTimeline, setShowTimeline] = useState(false);

  const formatBudget = (min?: number, max?: number) => {
    if (!min && !max) return 'Not specified';
    if (min && max) return `₹${(min / 100000).toFixed(1)}L - ₹${(max / 100000).toFixed(1)}L`;
    if (min) return `₹${(min / 100000).toFixed(1)}L+`;
    return `Up to ₹${(max! / 100000).toFixed(1)}L`;
  };

  const getAIScoreBadge = (score: number) => {
    if (score >= 80) return { color: 'bg-green-100 text-green-800', label: 'Hot' };
    if (score >= 60) return { color: 'bg-yellow-100 text-yellow-800', label: 'Warm' };
    return { color: 'bg-gray-100 text-gray-800', label: 'Cold' };
  };

  const getSLAStatus = () => {
    if (!lead.sla_deadline) return null;
    const deadline = new Date(lead.sla_deadline);
    const now = new Date();
    const hoursLeft = (deadline.getTime() - now.getTime()) / (1000 * 60 * 60);
    
    if (lead.is_overdue) {
      return { color: 'text-red-600', text: 'Overdue', urgent: true };
    } else if (hoursLeft <= 2) {
      return { color: 'text-orange-600', text: `${Math.ceil(hoursLeft)}h left`, urgent: true };
    } else if (hoursLeft <= 24) {
      return { color: 'text-yellow-600', text: `${Math.ceil(hoursLeft)}h left`, urgent: false };
    }
    return { color: 'text-green-600', text: `${Math.ceil(hoursLeft / 24)}d left`, urgent: false };
  };

  const aiScoreBadge = getAIScoreBadge(lead.ai_score);
  const slaStatus = getSLAStatus();

  return (
    <>
      <div
        className={`bg-white rounded-lg shadow-sm border border-gray-200 p-3 cursor-move hover:shadow-md transition-all duration-200 ${
          isDragging ? 'shadow-lg rotate-2 scale-105' : ''
        } ${lead.is_overdue ? 'border-l-4 border-l-red-500' : ''} ${slaStatus?.urgent ? 'ring-2 ring-red-200' : ''}`}
      >
        {/* Header */}
        <div className="flex items-start justify-between mb-2">
          <div className="flex-1 min-w-0">
            <h3 className="font-semibold text-gray-900 text-sm truncate">{lead.buyer_name}</h3>
            <p className="text-xs text-gray-500 truncate">{lead.property_name || `Property ${lead.property_id}`}</p>
          </div>
          <div className="flex items-center gap-1 ml-2">
            <div className={`px-2 py-1 rounded-full text-xs font-bold ${aiScoreBadge.color}`}>
              {lead.ai_score}
            </div>
            {slaStatus?.urgent && <AlertCircle size={12} className="text-red-500" />}
          </div>
        </div>

        {/* Budget & Location */}
        <div className="mb-2">
          <div className="flex items-center gap-1 mb-1">
            <DollarSign size={12} className="text-green-600" />
            <span className="text-xs font-medium text-gray-900">{formatBudget(lead.budget_min, lead.budget_max)}</span>
          </div>
          {lead.city && (
            <div className="flex items-center gap-1">
              <MapPin size={12} className="text-blue-600" />
              <span className="text-xs text-gray-600">{lead.city}</span>
            </div>
          )}
        </div>

        {/* Contact & Actions */}
        <div className="flex items-center justify-between text-xs">
          <div className="flex items-center gap-2">
            <button
              onClick={(e) => {
                e.stopPropagation();
                setShowTimeline(true);
              }}
              className="flex items-center gap-1 text-blue-600 hover:text-blue-800 transition-colors"
            >
              <Clock size={12} />
              <span>Timeline</span>
            </button>
            {lead.buyer_phone && (
              <button className="flex items-center gap-1 text-green-600 hover:text-green-800 transition-colors">
                <Phone size={12} />
              </button>
            )}
            <button className="flex items-center gap-1 text-purple-600 hover:text-purple-800 transition-colors">
              <Mail size={12} />
            </button>
          </div>
          <div className="flex items-center gap-1">
            <span className={`px-2 py-1 rounded-full text-xs font-medium ${
              lead.priority === 'High' ? 'bg-red-100 text-red-700' :
              lead.priority === 'Medium' ? 'bg-yellow-100 text-yellow-700' :
              'bg-gray-100 text-gray-700'
            }`}>
              {lead.priority}
            </span>
          </div>
        </div>
        
        {/* Source & Time */}
        <div className="mt-2 pt-2 border-t border-gray-100 flex items-center justify-between text-xs">
          <span className="text-gray-500">{lead.source}</span>
          <span className="text-gray-400">{formatDistanceToNow(new Date(lead.last_activity_at), { addSuffix: true })}</span>
        </div>
      </div>

      {/* Activity Timeline Modal */}
      {showTimeline && (
        <ActivityTimeline
          leadId={lead.id}
          leadName={lead.buyer_name}
          onClose={() => setShowTimeline(false)}
          onRefresh={onRefresh}
        />
      )}
    </>
  );
};

export default PipelineCard;
