import React from 'react';
import { PipelineFilters, PIPELINE_STAGES } from '../../types/pipeline';
import { Search, Filter } from 'lucide-react';

interface PipelineFiltersBarProps {
  filters: PipelineFilters;
  onFiltersChange: (filters: PipelineFilters) => void;
}

const PipelineFiltersBar: React.FC<PipelineFiltersBarProps> = ({ filters, onFiltersChange }) => {
  const handleFilterChange = (key: keyof PipelineFilters, value: any) => {
    onFiltersChange({ ...filters, [key]: value });
  };

  return (
    <div className="mt-4 flex flex-wrap gap-3 items-center">
      <div className="relative flex-1 min-w-[200px]">
        <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" size={18} />
        <input
          type="text"
          placeholder="Search by name, phone, or property..."
          className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
          value={filters.search || ''}
          onChange={(e) => handleFilterChange('search', e.target.value)}
        />
      </div>

      <select
        className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
        value={filters.stage || ''}
        onChange={(e) => handleFilterChange('stage', e.target.value || undefined)}
      >
        <option value="">All Stages</option>
        {PIPELINE_STAGES.map((stage) => (
          <option key={stage} value={stage}>{stage}</option>
        ))}
      </select>

      <select
        className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
        value={filters.source || ''}
        onChange={(e) => handleFilterChange('source', e.target.value || undefined)}
      >
        <option value="">All Sources</option>
        <option value="Website">Website</option>
        <option value="Campaign">Campaign</option>
        <option value="Referral">Referral</option>
        <option value="Agent">Agent</option>
        <option value="Social Media">Social Media</option>
        <option value="Direct">Direct</option>
      </select>

      <input
        type="text"
        placeholder="City"
        className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
        value={filters.city || ''}
        onChange={(e) => handleFilterChange('city', e.target.value || undefined)}
      />

      <div className="flex items-center gap-2">
        <input
          type="number"
          placeholder="Min Budget (L)"
          className="w-32 px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
          value={filters.budget_min ? filters.budget_min / 100000 : ''}
          onChange={(e) => handleFilterChange('budget_min', e.target.value ? parseFloat(e.target.value) * 100000 : undefined)}
        />
        <span className="text-gray-400">-</span>
        <input
          type="number"
          placeholder="Max Budget (L)"
          className="w-32 px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
          value={filters.budget_max ? filters.budget_max / 100000 : ''}
          onChange={(e) => handleFilterChange('budget_max', e.target.value ? parseFloat(e.target.value) * 100000 : undefined)}
        />
      </div>

      <label className="flex items-center gap-2 px-4 py-2 border border-gray-300 rounded-lg cursor-pointer hover:bg-gray-50">
        <input
          type="checkbox"
          checked={filters.is_overdue || false}
          onChange={(e) => handleFilterChange('is_overdue', e.target.checked || undefined)}
          className="rounded text-blue-600 focus:ring-2 focus:ring-blue-500"
        />
        <span className="text-sm text-gray-700">Overdue Only</span>
      </label>

      <button
        onClick={() => onFiltersChange({})}
        className="px-4 py-2 text-sm text-gray-600 hover:text-gray-900 hover:bg-gray-100 rounded-lg transition-colors"
      >
        Clear Filters
      </button>
    </div>
  );
};

export default PipelineFiltersBar;
