import React, { useState } from 'react';
import { MarketInsightsResponse, getMarketAnalysis } from '../../api/price-prediction/pricePrediction';
import { LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, Area, AreaChart } from 'recharts';

interface PSFForecastingProps {
    onError?: (error: string) => void;
}

export const PSFForecasting: React.FC<PSFForecastingProps> = ({ onError }) => {
    const [location, setLocation] = useState('');
    const [loading, setLoading] = useState(false);
    const [insights, setInsights] = useState<MarketInsightsResponse | null>(null);

    const showError = (message: string) => {
        if (onError) {
            onError(message);
        } else {
            alert(message); // Fallback if no error handler provided
        }
    };

    const showSuccess = (message: string) => {
        // You can implement a success notification here
        console.log(message);
    };

    const handleSearch = async (e: React.FormEvent) => {
        e.preventDefault();

        if (!location.trim()) {
            showError('Please enter a location');
            return;
        }

        setLoading(true);
        try {
            const result = await getMarketAnalysis(location);
            setInsights(result);
            showSuccess('PSF forecasting loaded!');
        } catch (error: any) {
            showError(error.message || 'Failed to load PSF forecasting');
            console.error('PSF forecasting error:', error);
        } finally {
            setLoading(false);
        }
    };

    const prepareChartData = () => {
        if (!insights?.price_trend_2020_2030) return [];
        
        const data = Object.entries(insights.price_trend_2020_2030)
            .map(([year, price]) => ({
                year: year,
                price: Number(price)
            }))
            .sort((a, b) => Number(a.year) - Number(b.year));
        
        return data;
    };

    const formatYAxis = (value: number) => {
        if (value >= 10000) {
            return `₹${(value / 1000).toFixed(0)}K`;
        }
        return `₹${value}`;
    };

    const CustomTooltip = ({ active, payload }: any) => {
        if (active && payload && payload.length) {
            return (
                <div className="bg-white px-4 py-2 rounded-lg shadow-lg border border-gray-200">
                    <p className="text-sm font-semibold text-gray-900">Year: {payload[0].payload.year}</p>
                    <p className="text-sm text-blue-600 font-bold">₹{payload[0].value.toLocaleString('en-IN')} PSF</p>
                </div>
            );
        }
        return null;
    };

    return (
        <div className="space-y-6">
            {/* Search Form */}
            <form onSubmit={handleSearch} className="bg-white p-6 rounded-xl shadow-lg border border-gray-100">
                <h3 className="text-xl font-bold text-gray-900 mb-4 flex items-center gap-2">
                    <svg className="w-6 h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
                    </svg>
                    PSF Forecasting
                </h3>
                <div className="flex gap-3">
                    <input
                        type="text"
                        value={location}
                        onChange={(e) => setLocation(e.target.value)}
                        placeholder="Enter location (e.g., Indiranagar, Koramangala)"
                        className="flex-1 px-4 py-3 rounded-xl border border-gray-200 focus:border-[#0056D2] focus:ring-2 focus:ring-blue-100 outline-none transition-all"
                    />
                    <button
                        type="submit"
                        disabled={loading}
                        className="bg-[#0056D2] hover:bg-[#0056D2]/90 text-white font-semibold px-8 py-3 rounded-xl transition-all duration-300 shadow-lg hover:shadow-xl disabled:opacity-50 disabled:cursor-not-allowed"
                    >
                        {loading ? 'Analyzing...' : 'Analyze'}
                    </button>
                </div>
            </form>

            {/* Results */}
            {insights && (
                <div className="space-y-6">
                    {/* Location Info */}
                    <div className="bg-[#0056D2] p-6 rounded-xl shadow-lg text-white">
                        <h4 className="text-2xl font-bold mb-2">{insights.location}</h4>
                        {insights.zone && (
                            <p className="text-blue-100">Zone: {insights.zone}</p>
                        )}
                        {insights.price_trend_2020_2030 && insights.price_trend_2020_2030['2025'] && (
                            <div className="mt-4 pt-4 border-t border-white/20">
                                <p className="text-sm text-blue-100 mb-1">Average PSF (2025)</p>
                                <p className="text-3xl font-bold">₹{Number(insights.price_trend_2020_2030['2025']).toLocaleString('en-IN')}</p>
                            </div>
                        )}
                    </div>

                    {/* Key Metrics */}
                    {(insights.growth_rate_2025 || insights.investment_score || insights.demand_level) && (
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                            {insights.growth_rate_2025 !== undefined && (
                                <div className="bg-green-50 p-6 rounded-xl border border-green-200">
                                    <div className="flex items-center gap-2 mb-2">
                                        <svg className="w-5 h-5 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6" />
                                        </svg>
                                        <span className="text-sm font-semibold text-gray-700">Growth Rate</span>
                                    </div>
                                    <p className="text-2xl font-bold text-green-600">{insights.growth_rate_2025.toFixed(2)}%</p>
                                </div>
                            )}

                            {insights.investment_score && (
                                <div className="bg-blue-50 p-6 rounded-xl border border-blue-200">
                                    <div className="flex items-center gap-2 mb-2">
                                        <svg className="w-5 h-5 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M11.049 2.927c.3-.921 1.603-.921 1.902 0l1.519 4.674a1 1 0 00.95.69h4.915c.969 0 1.371 1.24.588 1.81l-3.976 2.888a1 1 0 00-.363 1.118l1.518 4.674c.3.922-.755 1.688-1.538 1.118l-3.976-2.888a1 1 0 00-1.176 0l-3.976 2.888c-.783.57-1.838-.197-1.538-1.118l1.518-4.674a1 1 0 00-.363-1.118l-3.976-2.888c-.784-.57-.38-1.81.588-1.81h4.914a1 1 0 00.951-.69l1.519-4.674z" />
                                        </svg>
                                        <span className="text-sm font-semibold text-gray-700">Investment Score</span>
                                    </div>
                                    <p className="text-2xl font-bold text-[#0056D2]">{insights.investment_score.toFixed(1)}/10</p>
                                </div>
                            )}

                            {insights.demand_level && (
                                <div className="bg-purple-50 p-6 rounded-xl border border-purple-200">
                                    <div className="flex items-center gap-2 mb-2">
                                        <svg className="w-5 h-5 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z" />
                                        </svg>
                                        <span className="text-sm font-semibold text-gray-700">Demand Level</span>
                                    </div>
                                    <p className="text-2xl font-bold text-purple-600">{insights.demand_level}</p>
                                </div>
                            )}
                        </div>
                    )}

                    {/* Price Trends 2020-2030 - Graph */}
                    {insights.price_trend_2020_2030 && Object.keys(insights.price_trend_2020_2030).length > 0 && (
                        <div className="bg-white p-6 rounded-xl shadow-lg border border-gray-100">
                            <h4 className="text-lg font-semibold text-gray-900 mb-6 flex items-center gap-2">
                                <svg className="w-5 h-5 text-[#0056D2]" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M7 12l3-3 3 3 4-4M8 21l4-4 4 4M3 4h18M4 4h16v12a1 1 0 01-1 1H5a1 1 0 01-1-1V4z" />
                                </svg>
                                Price Trends (2020-2030)
                            </h4>
                            <div className="w-full" style={{ height: '400px' }}>
                                <ResponsiveContainer width="100%" height="100%">
                                    <AreaChart
                                        data={prepareChartData()}
                                        margin={{ top: 10, right: 30, left: 0, bottom: 0 }}
                                    >
                                        <defs>
                                            <linearGradient id="colorPrice" x1="0" y1="0" x2="0" y2="1">
                                                <stop offset="5%" stopColor="#0056D2" stopOpacity={0.3}/>
                                                <stop offset="95%" stopColor="#0056D2" stopOpacity={0}/>
                                            </linearGradient>
                                        </defs>
                                        <CartesianGrid strokeDasharray="3 3" stroke="#e5e7eb" />
                                        <XAxis 
                                            dataKey="year" 
                                            stroke="#6b7280"
                                            style={{ fontSize: '12px', fontWeight: '500' }}
                                            tick={{ fill: '#374151' }}
                                        />
                                        <YAxis 
                                            tickFormatter={formatYAxis}
                                            stroke="#6b7280"
                                            style={{ fontSize: '12px', fontWeight: '500' }}
                                            tick={{ fill: '#374151' }}
                                        />
                                        <Tooltip content={<CustomTooltip />} />
                                        <Area 
                                            type="monotone" 
                                            dataKey="price" 
                                            stroke="#0056D2" 
                                            strokeWidth={3}
                                            fill="url(#colorPrice)"
                                            animationDuration={1500}
                                        />
                                    </AreaChart>
                                </ResponsiveContainer>
                            </div>
                            <div className="mt-4 flex items-center justify-center gap-2 text-sm text-gray-600">
                                <div className="flex items-center gap-1">
                                    <div className="w-3 h-3 rounded-full bg-[#0056D2]"></div>
                                    <span>Price per Sq.Ft (PSF)</span>
                                </div>
                            </div>
                        </div>
                    )}

                    {/* Key Insights */}
                    {insights.key_insights && insights.key_insights.length > 0 && (
                        <div className="bg-yellow-50 p-6 rounded-xl border border-yellow-200">
                            <h4 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
                                <svg className="w-5 h-5 text-yellow-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9.663 17h4.673M12 3v1m6.364 1.636l-.707.707M21 12h-1M4 12H3m3.343-5.657l-.707-.707m2.828 9.9a5 5 0 117.072 0l-.548.547A3.374 3.374 0 0014 18.469V19a2 2 0 11-4 0v-.531c0-.895-.356-1.754-.988-2.386l-.548-.547z" />
                                </svg>
                                Key Insights
                            </h4>
                            <div className="space-y-2">
                                {insights.key_insights.map((insight, idx) => (
                                    <div key={idx} className="flex items-start gap-2">
                                        <span className="text-yellow-600 mt-1">💡</span>
                                        <span className="text-sm text-gray-700">{insight}</span>
                                    </div>
                                ))}
                            </div>
                        </div>
                    )}
                </div>
            )}

            {/* Empty State */}
            {!insights && !loading && (
                <div className="bg-gray-50 p-12 rounded-xl border-2 border-dashed border-gray-300 text-center">
                    <svg className="w-16 h-16 mx-auto text-gray-400 mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
                    </svg>
                    <p className="text-gray-600 font-medium">Enter a location to view PSF forecasting</p>
                    <p className="text-sm text-gray-500 mt-1">Get insights on price trends, growth rates, and investment potential</p>
                </div>
            )}
        </div>
    );
};

export default PSFForecasting;