import React from 'react';
import { PriceBreakdown as PriceBreakdownType } from '../../api/price-prediction/pricePrediction';

interface PriceBreakdownProps {
    breakdown: PriceBreakdownType;
}

export const PriceBreakdown: React.FC<PriceBreakdownProps> = ({ breakdown }) => {
    const formatPercentage = (value: number) => {
        const percentage = (value * 100).toFixed(2);
        return value >= 0 ? `+${percentage}%` : `${percentage}%`;
    };

    const formatMultiplier = (value: number) => {
        return `${value.toFixed(2)}x`;
    };

    const adjustments = [
        {
            label: 'Age Factor',
            value: breakdown.age_factor,
            icon: '🏗️',
            color: 'text-purple-600',
            bgColor: 'bg-purple-50',
            isMultiplier: true
        },
        {
            label: 'Amenity Bonus',
            value: breakdown.amenity_bonus,
            icon: '🎯',
            color: 'text-green-600',
            bgColor: 'bg-green-50'
        },
        {
            label: 'Furnishing Bonus',
            value: breakdown.furnishing_bonus,
            icon: '🛋️',
            color: 'text-orange-600',
            bgColor: 'bg-orange-50'
        },
        {
            label: 'Floor Premium',
            value: breakdown.floor_premium,
            icon: '🏢',
            color: 'text-indigo-600',
            bgColor: 'bg-indigo-50'
        },
        {
            label: 'Listing Type Adjustment',
            value: breakdown.listing_type_adjustment,
            icon: '💼',
            color: 'text-pink-600',
            bgColor: 'bg-pink-50'
        }
    ];

    return (
        <div className="bg-white p-6 rounded-xl shadow-lg border border-gray-100">
            <h4 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
                <svg className="w-5 h-5 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 7h6m0 10v-3m-3 3h.01M9 17h.01M9 14h.01M12 14h.01M15 11h.01M12 11h.01M9 11h.01M7 21h10a2 2 0 002-2V5a2 2 0 00-2-2H7a2 2 0 00-2 2v14a2 2 0 002 2z" />
                </svg>
                Price Breakdown
            </h4>

            {/* Base Price */}
            <div className="mb-6 p-4 bg-gradient-to-r from-gray-50 to-gray-100 rounded-lg border border-gray-200">
                <div className="flex items-center justify-between">
                    <span className="text-sm font-semibold text-gray-700">Base Price</span>
                    <span className="text-xl font-bold text-gray-900">
                        ₹{breakdown.base_price.toLocaleString('en-IN')}
                    </span>
                </div>
            </div>

            {/* Adjustments */}
            <div className="space-y-3">
                {adjustments.map((adj, idx) => (
                    <div
                        key={idx}
                        className={`p-4 rounded-lg ${adj.bgColor} border border-gray-200 transition-all hover:shadow-md`}
                    >
                        <div className="flex items-center justify-between">
                            <div className="flex items-center gap-3">
                                <span className="text-2xl">{adj.icon}</span>
                                <span className="text-sm font-medium text-gray-700">{adj.label}</span>
                            </div>
                            <div className="flex items-center gap-2">
                                <span className={`text-lg font-bold ${adj.color}`}>
                                    {adj.isMultiplier ? formatMultiplier(adj.value) : formatPercentage(adj.value)}
                                </span>
                                {Math.abs(adj.value) > 0 && (
                                    <span className="text-xs">
                                        {adj.value > 0 ? '📈' : adj.value < 0 ? '📉' : '➖'}
                                    </span>
                                )}
                            </div>
                        </div>

                        {/* Progress bar for percentage adjustments */}
                        {!adj.isMultiplier && Math.abs(adj.value) > 0 && (
                            <div className="mt-2">
                                <div className="w-full bg-gray-200 rounded-full h-1.5">
                                    <div
                                        className={`h-1.5 rounded-full transition-all duration-500 ${
                                            adj.value > 0 ? 'bg-green-500' : 'bg-red-500'
                                        }`}
                                        style={{ width: `${Math.min(Math.abs(adj.value) * 100, 100)}%` }}
                                    />
                                </div>
                            </div>
                        )}
                    </div>
                ))}
            </div>

            {/* Info Note */}
            <div className="mt-6 p-3 bg-blue-50 rounded-lg border border-blue-200">
                <p className="text-xs text-blue-800 flex items-start gap-2">
                    <svg className="w-4 h-4 mt-0.5 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                        <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clipRule="evenodd" />
                    </svg>
                    <span>
                        Base price is calculated using Area × Year PSF from forecasting data. Adjustments are then applied for property features like age, amenities, furnishing, and floor position.
                    </span>
                </p>
            </div>
        </div>
    );
};

export default PriceBreakdown;
