import React, { useState } from 'react';
import {
    PropertyType,
    ListingType,
    PredictionRequest,
    PredictionResponse,
    predictPrice
} from '../../api/price-prediction/pricePrediction';

interface PricePredictionFormProps {
    onPredictionComplete?: (result: PredictionResponse) => void;
    onError?: (error: string) => void;
}

const AMENITIES_OPTIONS = [
    'Gym', 'Swimming Pool', 'Parking', 'Lift', 'Security', 'Power Backup',
    'Garden', 'Children Play Area', 'Club House', 'Intercom', 'Gas Pipeline',
    'Wifi', 'Air Conditioning', 'Servant Room', 'Rain Water Harvesting'
];

// Area guidelines based on BHK configuration
const getAreaGuidelines = (bedrooms: number) => {
    const guidelines: Record<number, { min: number; max: number; typical: number }> = {
        1: { min: 400, max: 700, typical: 550 },
        2: { min: 700, max: 1200, typical: 900 },
        3: { min: 1100, max: 1800, typical: 1400 },
        4: { min: 1600, max: 2500, typical: 2000 },
        5: { min: 2200, max: 3500, typical: 2800 },
    };
    return guidelines[bedrooms] || { min: 500, max: 5000, typical: 1500 };
};

export const PricePredictionForm: React.FC<PricePredictionFormProps> = ({ 
    onPredictionComplete, 
    onError 
}) => {
    const [loading, setLoading] = useState(false);
    const [formData, setFormData] = useState<PredictionRequest>({
        property_type: PropertyType.APARTMENT,
        listing_type: ListingType.SELL,
        bedrooms: 2,
        bathrooms: 2,
        balconies: 1,
        built_up_area: 1000,
        carpet_area: 800,
        locality: '',
        furnishing: 'Semi Furnished',
        age_years: 0,
        amenities: [],
        floor_number: 1,
        total_floors: 10,
        query: '',
        prediction_year: 2025
    });

    const areaGuidelines = getAreaGuidelines(formData.bedrooms);

    const handleInputChange = (field: keyof PredictionRequest, value: any) => {
        setFormData(prev => {
            const updated = { ...prev, [field]: value };
            
            // Auto-adjust area when bedrooms change
            if (field === 'bedrooms') {
                const guidelines = getAreaGuidelines(value);
                // Only adjust if current area is outside reasonable range
                if (prev.built_up_area < guidelines.min || prev.built_up_area > guidelines.max) {
                    updated.built_up_area = guidelines.typical;
                    updated.carpet_area = Math.round(guidelines.typical * 0.75);
                }
            }
            
            // Auto-adjust carpet area when built-up area changes
            if (field === 'built_up_area' && value > 0) {
                updated.carpet_area = Math.round(value * 0.75);
            }
            
            return updated;
        });
    };

    const handleAmenityToggle = (amenity: string) => {
        setFormData(prev => ({
            ...prev,
            amenities: prev.amenities?.includes(amenity)
                ? prev.amenities.filter(a => a !== amenity)
                : [...(prev.amenities || []), amenity]
        }));
    };

    const showError = (message: string) => {
        if (onError) {
            onError(message);
        } else {
            alert(message); // Fallback if no error handler provided
        }
    };

    const showSuccess = (message: string) => {
        // You can implement a success notification here
        console.log(message);
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();

        if (!formData.locality) {
            showError('Please enter a locality');
            return;
        }

        // Validate area based on bedrooms
        const guidelines = getAreaGuidelines(formData.bedrooms);
        if (formData.built_up_area < guidelines.min) {
            showError(`Built-up area too small for ${formData.bedrooms}BHK. Minimum: ${guidelines.min} sqft`);
            return;
        }
        if (formData.built_up_area > guidelines.max) {
            console.warn(`Built-up area seems large for ${formData.bedrooms}BHK. Typical range: ${guidelines.min}-${guidelines.max} sqft`);
        }

        setLoading(true);
        
        // Add a safety timeout to prevent infinite loading
        const safetyTimeout = setTimeout(() => {
            setLoading(false);
            showError('Request is taking too long. Please check if the backend server is running and try again.');
        }, 35000); // 35 seconds safety timeout
        
        try {
            const result = await predictPrice(formData);
            clearTimeout(safetyTimeout);
            showSuccess('Price prediction completed!');
            onPredictionComplete?.(result);
        } catch (error: any) {
            clearTimeout(safetyTimeout);
            const errorMessage = error.message || 'Failed to predict price';
            
            // Check if it's a network/timeout error
            if (errorMessage.includes('timeout') || errorMessage.includes('fetch')) {
                showError('Unable to connect to the prediction service. Please ensure the backend server is running on http://localhost:8000');
            } else {
                showError(errorMessage);
            }
            
            console.error('Prediction error:', error);
        } finally {
            clearTimeout(safetyTimeout);
            setLoading(false);
        }
    };

    return (
        <form onSubmit={handleSubmit} className="space-y-6 bg-white p-8 rounded-2xl shadow-lg border border-gray-100">
            <div className="space-y-2">
                <h2 className="text-2xl font-bold text-gray-900">Property Details</h2>
                <p className="text-sm text-gray-600">Fill in the details to get an AI-powered price prediction</p>
            </div>

            {/* Property Type */}
            <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">Property Type</label>
                <div className="grid grid-cols-2 sm:grid-cols-3 gap-3">
                    {Object.values(PropertyType).map(type => (
                        <button
                            key={type}
                            type="button"
                            onClick={() => handleInputChange('property_type', type)}
                            className={`px-4 py-3 rounded-xl text-sm font-bold transition-all duration-300 ${
                                formData.property_type === type
                                    ? 'bg-[#0056D2] text-white shadow-lg scale-105 border-2 border-[#0056D2]'
                                    : 'bg-white text-gray-700 hover:bg-blue-50 hover:text-[#0056D2] border-2 border-gray-200 hover:border-[#0056D2] hover:shadow-md'
                            }`}
                        >
                            {type}
                        </button>
                    ))}
                </div>
            </div>

            {/* Listing Type */}
            <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">Listing Type</label>
                <div className="grid grid-cols-2 sm:grid-cols-3 gap-3">
                    {Object.values(ListingType).map(type => (
                        <button
                            key={type}
                            type="button"
                            onClick={() => handleInputChange('listing_type', type)}
                            className={`px-4 py-3 rounded-xl text-sm font-bold transition-all duration-300 ${
                                formData.listing_type === type
                                    ? 'bg-[#0056D2] text-white shadow-lg scale-105 border-2 border-[#0056D2]'
                                    : 'bg-white text-gray-700 hover:bg-blue-50 hover:text-[#0056D2] border-2 border-gray-200 hover:border-[#0056D2] hover:shadow-md'
                            }`}
                        >
                            {type}
                        </button>
                    ))}
                </div>
            </div>

            {/* Basic Details - Grid */}
            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                <div>
                    <label className="block text-sm font-semibold text-gray-700 mb-2">Bedrooms</label>
                    <input
                        type="number"
                        min="1"
                        max="10"
                        value={formData.bedrooms}
                        onChange={(e) => handleInputChange('bedrooms', parseInt(e.target.value))}
                        className="w-full px-4 py-3 rounded-xl border border-gray-200 focus:border-blue-500 focus:ring-2 focus:ring-blue-200 outline-none transition-all"
                        required
                    />
                </div>

                <div>
                    <label className="block text-sm font-semibold text-gray-700 mb-2">Bathrooms</label>
                    <input
                        type="number"
                        min="1"
                        max="10"
                        value={formData.bathrooms}
                        onChange={(e) => handleInputChange('bathrooms', parseInt(e.target.value))}
                        className="w-full px-4 py-3 rounded-xl border border-gray-200 focus:border-blue-500 focus:ring-2 focus:ring-blue-200 outline-none transition-all"
                        required
                    />
                </div>

                <div>
                    <label className="block text-sm font-semibold text-gray-700 mb-2">Balconies</label>
                    <input
                        type="number"
                        min="0"
                        max="10"
                        value={formData.balconies}
                        onChange={(e) => handleInputChange('balconies', parseInt(e.target.value))}
                        className="w-full px-4 py-3 rounded-xl border border-gray-200 focus:border-blue-500 focus:ring-2 focus:ring-blue-200 outline-none transition-all"
                    />
                </div>

                <div>
                    <label className="block text-sm font-semibold text-gray-700 mb-2">Age (Years)</label>
                    <input
                        type="number"
                        min="0"
                        max="100"
                        value={formData.age_years}
                        onChange={(e) => handleInputChange('age_years', parseInt(e.target.value))}
                        className="w-full px-4 py-3 rounded-xl border border-gray-200 focus:border-blue-500 focus:ring-2 focus:ring-blue-200 outline-none transition-all"
                    />
                </div>
            </div>

            {/* Area Details */}
            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                <div>
                    <label className="block text-sm font-semibold text-gray-700 mb-2">Built-up Area (sqft) *</label>
                    <input
                        type="number"
                        min={areaGuidelines.min}
                        max={areaGuidelines.max}
                        value={formData.built_up_area}
                        onChange={(e) => handleInputChange('built_up_area', parseFloat(e.target.value))}
                        className={`w-full px-4 py-3 rounded-xl border focus:ring-2 outline-none transition-all ${
                            formData.built_up_area < areaGuidelines.min || formData.built_up_area > areaGuidelines.max
                                ? 'border-orange-300 focus:border-orange-500 focus:ring-orange-200'
                                : 'border-gray-200 focus:border-blue-500 focus:ring-blue-200'
                        }`}
                        required
                    />
                    <p className="text-xs text-gray-500 mt-1">
                        Recommended: {areaGuidelines.min}-{areaGuidelines.max} sqft for {formData.bedrooms}BHK
                    </p>
                </div>

                <div>
                    <label className="block text-sm font-semibold text-gray-700 mb-2">Carpet Area (sqft)</label>
                    <input
                        type="number"
                        min="100"
                        value={formData.carpet_area}
                        onChange={(e) => handleInputChange('carpet_area', parseFloat(e.target.value))}
                        className="w-full px-4 py-3 rounded-xl border border-gray-200 focus:border-blue-500 focus:ring-2 focus:ring-blue-200 outline-none transition-all"
                    />
                    <p className="text-xs text-gray-500 mt-1">
                        Typically 70-80% of built-up area
                    </p>
                </div>
            </div>

            {/* Locality */}
            <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">Locality *</label>
                <input
                    type="text"
                    value={formData.locality}
                    onChange={(e) => handleInputChange('locality', e.target.value)}
                    placeholder="e.g., Indiranagar, Koramangala, Whitefield"
                    className="w-full px-4 py-3 rounded-xl border border-gray-200 focus:border-blue-500 focus:ring-2 focus:ring-blue-200 outline-none transition-all"
                    required
                />
            </div>

            {/* Prediction Year */}
            <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">Prediction Year</label>
                <div className="grid grid-cols-3 sm:grid-cols-6 gap-2">
                    {[2025, 2026, 2027, 2028, 2029, 2030].map(year => (
                        <button
                            key={year}
                            type="button"
                            onClick={() => handleInputChange('prediction_year', year)}
                            className={`px-3 py-2 rounded-lg text-sm font-bold transition-all duration-300 ${
                                formData.prediction_year === year
                                    ? 'bg-[#0056D2] text-white shadow-lg scale-105 border-2 border-[#0056D2]'
                                    : 'bg-white text-gray-700 hover:bg-blue-50 hover:text-[#0056D2] border-2 border-gray-200 hover:border-[#0056D2] hover:shadow-md'
                            }`}
                        >
                            {year}
                        </button>
                    ))}
                </div>
                <p className="text-xs text-gray-500 mt-2">
                    Select the year for which you want to predict the property price
                </p>
            </div>

            {/* Furnishing */}
            <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">Furnishing Status</label>
                <div className="grid grid-cols-3 gap-3">
                    {['Fully Furnished', 'Semi Furnished', 'Unfurnished'].map(status => (
                        <button
                            key={status}
                            type="button"
                            onClick={() => handleInputChange('furnishing', status)}
                            className={`px-4 py-3 rounded-xl text-sm font-bold transition-all duration-300 ${
                                formData.furnishing === status
                                    ? 'bg-[#0056D2] text-white shadow-lg scale-105 border-2 border-[#0056D2]'
                                    : 'bg-white text-gray-700 hover:bg-blue-50 hover:text-[#0056D2] border-2 border-gray-200 hover:border-[#0056D2] hover:shadow-md'
                            }`}
                        >
                            {status}
                        </button>
                    ))}
                </div>
            </div>

            {/* Floor Details */}
            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                <div>
                    <label className="block text-sm font-semibold text-gray-700 mb-2">Floor Number</label>
                    <input
                        type="number"
                        min="0"
                        value={formData.floor_number}
                        onChange={(e) => handleInputChange('floor_number', parseInt(e.target.value))}
                        className="w-full px-4 py-3 rounded-xl border border-gray-200 focus:border-blue-500 focus:ring-2 focus:ring-blue-200 outline-none transition-all"
                    />
                </div>

                <div>
                    <label className="block text-sm font-semibold text-gray-700 mb-2">Total Floors</label>
                    <input
                        type="number"
                        min="0"
                        value={formData.total_floors}
                        onChange={(e) => handleInputChange('total_floors', parseInt(e.target.value))}
                        className="w-full px-4 py-3 rounded-xl border border-gray-200 focus:border-blue-500 focus:ring-2 focus:ring-blue-200 outline-none transition-all"
                    />
                </div>
            </div>

            {/* Amenities */}
            <div>
                <label className="block text-sm font-semibold text-gray-700 mb-3">
                    Amenities ({formData.amenities?.length || 0} selected)
                </label>
                <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 gap-2 max-h-48 overflow-y-auto p-1">
                    {AMENITIES_OPTIONS.map(amenity => (
                        <button
                            key={amenity}
                            type="button"
                            onClick={() => handleAmenityToggle(amenity)}
                            className={`px-3 py-2 rounded-lg text-xs font-medium transition-all ${
                                formData.amenities?.includes(amenity)
                                    ? 'bg-[#0056D2] text-white border-2 border-[#0056D2]'
                                    : 'bg-gray-50 text-gray-600 hover:bg-blue-50 hover:text-[#0056D2] border border-gray-200'
                            }`}
                        >
                            {amenity}
                        </button>
                    ))}
                </div>
            </div>

            {/* Submit Button */}
            <button
                type="submit"
                disabled={loading}
                className="w-full bg-[#0056D2] hover:bg-[#0056D2]/90 text-white font-semibold py-4 px-6 rounded-xl transition-all duration-300 shadow-lg hover:shadow-xl disabled:opacity-50 disabled:cursor-not-allowed flex items-center justify-center gap-2"
            >
                {loading ? (
                    <>
                        <svg className="animate-spin h-5 w-5" viewBox="0 0 24 24">
                            <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" fill="none" />
                            <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z" />
                        </svg>
                        <span>Analyzing... (This may take up to 30 seconds)</span>
                    </>
                ) : (
                    <>
                        <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 7h6m0 10v-3m-3 3h.01M9 17h.01M9 14h.01M12 14h.01M15 11h.01M12 11h.01M9 11h.01M7 21h10a2 2 0 002-2V5a2 2 0 00-2-2H7a2 2 0 00-2 2v14a2 2 0 002 2z" />
                        </svg>
                        <span>Get Price Prediction</span>
                    </>
                )}
            </button>
        </form>
    );
};

export default PricePredictionForm;
