import React from 'react';
import { PredictionResponse, formatCurrency, getConfidenceColor } from '../../api/price-prediction/pricePrediction';
import { PriceBreakdown } from './PriceBreakdown';

interface PricePredictionResultProps {
    result: PredictionResponse;
}

export const PricePredictionResult: React.FC<PricePredictionResultProps> = ({ result }) => {
    const confidencePercentage = (result.confidence_score * 100).toFixed(1);
    const confidenceColor = getConfidenceColor(result.confidence_score);

    return (
        <div className="space-y-6">
            {/* Main Price Card */}
            <div className="bg-[#0056D2] p-8 rounded-2xl shadow-2xl text-white">
                <div className="flex items-center justify-between mb-4">
                    <div className="flex items-center gap-2">
                        <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                        </svg>
                        <h3 className="text-lg font-semibold">Predicted Price for {result.prediction_year}</h3>
                    </div>
                    <div className="bg-white/20 backdrop-blur-sm px-3 py-1 rounded-full text-xs font-medium">
                        {result.model_version}
                    </div>
                </div>

                <div className="space-y-2">
                    <div className="text-5xl font-bold tracking-tight">
                        {formatCurrency(result.predicted_price, result.listing_type)}
                    </div>

                    <div className="flex items-center gap-4 text-sm text-white/90">
                        <div className="flex items-center gap-1">
                            <span>Range:</span>
                            <span className="font-semibold">{formatCurrency(result.price_range.min, result.listing_type)}</span>
                            <span>-</span>
                            <span className="font-semibold">{formatCurrency(result.price_range.max, result.listing_type)}</span>
                        </div>
                    </div>

                    {result.psf_used && (
                        <div className="flex items-center gap-2 text-sm text-white/90 mt-2">
                            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6" />
                            </svg>
                            <span>PSF for {result.prediction_year}: ₹{result.psf_used.toLocaleString('en-IN')}</span>
                        </div>
                    )}
                </div>

                {/* Confidence Score */}
                <div className="mt-6 pt-6 border-t border-white/20">
                    <div className="flex items-center justify-between mb-2">
                        <span className="text-sm text-white/90">Confidence Score</span>
                        <span className="text-lg font-bold">{confidencePercentage}%</span>
                    </div>
                    <div className="w-full bg-white/20 rounded-full h-2">
                        <div
                            className="bg-white h-2 rounded-full transition-all duration-500"
                            style={{ width: `${confidencePercentage}%` }}
                        />
                    </div>
                </div>
            </div>

            {/* Property Details Summary */}
            <div className="bg-white p-6 rounded-xl shadow-lg border border-gray-100">
                <h4 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
                    <svg className="w-5 h-5 text-[#0056D2]" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4" />
                    </svg>
                    Property Summary
                </h4>
                <div className="grid grid-cols-2 gap-4">
                    <div className="bg-blue-50 p-3 rounded-lg">
                        <span className="text-xs text-gray-600 block mb-1">Property Type</span>
                        <span className="text-sm font-bold text-[#0056D2]">{result.property_type}</span>
                    </div>
                    <div className="bg-blue-50 p-3 rounded-lg">
                        <span className="text-xs text-gray-600 block mb-1">Listing Type</span>
                        <span className="text-sm font-bold text-[#0056D2]">{result.listing_type}</span>
                    </div>
                    <div className="bg-blue-50 p-3 rounded-lg">
                        <span className="text-xs text-gray-600 block mb-1">Location</span>
                        <span className="text-sm font-bold text-[#0056D2] flex items-center gap-1">
                            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z" />
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 11a3 3 0 11-6 0 3 3 0 016 0z" />
                            </svg>
                            {result.locality}
                        </span>
                    </div>
                    <div className="bg-blue-50 p-3 rounded-lg">
                        <span className="text-xs text-gray-600 block mb-1">Prediction Year</span>
                        <span className="text-sm font-bold text-[#0056D2] flex items-center gap-1">
                            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" />
                            </svg>
                            {result.prediction_year}
                        </span>
                    </div>
                </div>
            </div>

            {/* Price Breakdown */}
            <PriceBreakdown breakdown={result.price_breakdown} />

            {/* Market Insights */}
            {result.market_insights && (
                <div className="bg-gradient-to-r from-green-50 to-blue-50 p-6 rounded-xl border border-green-200">
                    <h4 className="text-lg font-semibold text-gray-900 mb-3 flex items-center gap-2">
                        <svg className="w-5 h-5 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                        </svg>
                        Market Insights
                    </h4>
                    <p className="text-sm text-gray-700 leading-relaxed">{result.market_insights}</p>
                </div>
            )}

            {/* RAG Context */}
            {result.rag_context && result.rag_context.length > 0 && (
                <div className="bg-gray-50 p-6 rounded-xl border border-gray-200">
                    <h4 className="text-lg font-semibold text-gray-900 mb-3 flex items-center gap-2">
                        <svg className="w-5 h-5 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                        </svg>
                        Analysis Context
                    </h4>
                    <div className="space-y-2">
                        {result.rag_context.map((context, idx) => (
                            <div key={idx} className="text-xs text-gray-600 bg-white p-3 rounded-lg border border-gray-200">
                                {context}
                            </div>
                        ))}
                    </div>
                </div>
            )}

            {/* Timestamp */}
            <div className="text-xs text-gray-500 text-center">
                Predicted on {new Date(result.created_at).toLocaleString('en-IN')}
            </div>
        </div>
    );
};

export default PricePredictionResult;
