import React from 'react';
import { useBuyerInsights } from '../../hooks/pricing/usePricing';
import { Users, Eye, Heart, Calendar } from 'lucide-react';

interface Props {
  propertyId: number;
}

export const BuyerInsightsCard: React.FC<Props> = ({ propertyId }) => {
  const { data: insights, isLoading } = useBuyerInsights(propertyId);

  if (isLoading) {
    return <div className="animate-pulse bg-gray-200 h-64 rounded-lg"></div>;
  }

  if (!insights) {
    return null;
  }

  return (
    <div className="bg-white rounded-lg shadow-lg">
      <div className="p-6 border-b">
        <h3 className="text-lg font-bold flex items-center gap-2">
          <Users className="w-5 h-5" />
          Buyer Interest Analysis
        </h3>
      </div>

      <div className="p-6 space-y-6">
        {/* High-Intent Buyers Highlight */}
        {insights.high_intent_buyers > 0 && (
          <div className="bg-gradient-to-r from-green-50 to-emerald-50 p-4 rounded-lg border-2 border-green-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-3xl font-bold text-green-600">
                  {insights.high_intent_buyers}
                </p>
                <p className="text-sm text-gray-700">High-Intent Buyers</p>
              </div>
              {insights.high_intent_buyers >= 3 && (
                <span className="bg-green-600 text-white text-lg px-4 py-2 rounded-full font-semibold">
                  🔥 Strong Demand
                </span>
              )}
            </div>
          </div>
        )}

        {/* Engagement Metrics */}
        <div className="grid grid-cols-2 gap-4">
          <div className="flex items-center gap-3 p-3 bg-blue-50 rounded-lg">
            <Eye className="w-8 h-8 text-blue-600" />
            <div>
              <p className="text-2xl font-bold">{insights.metrics.total_views}</p>
              <p className="text-xs text-gray-600">Total Views</p>
            </div>
          </div>

          <div className="flex items-center gap-3 p-3 bg-purple-50 rounded-lg">
            <Heart className="w-8 h-8 text-purple-600" />
            <div>
              <p className="text-2xl font-bold">{insights.metrics.total_saves}</p>
              <p className="text-xs text-gray-600">Saves</p>
            </div>
          </div>

          <div className="flex items-center gap-3 p-3 bg-orange-50 rounded-lg">
            <Calendar className="w-8 h-8 text-orange-600" />
            <div>
              <p className="text-2xl font-bold">{insights.metrics.total_visits}</p>
              <p className="text-xs text-gray-600">Visits</p>
            </div>
          </div>

          <div className="flex items-center gap-3 p-3 bg-green-50 rounded-lg">
            <Users className="w-8 h-8 text-green-600" />
            <div>
              <p className="text-2xl font-bold">{insights.metrics.unique_users}</p>
              <p className="text-xs text-gray-600">Unique Users</p>
            </div>
          </div>
        </div>

        {/* Conversion Rates */}
        <div className="bg-gray-50 p-4 rounded-lg">
          <h4 className="font-semibold mb-3">Conversion Funnel</h4>
          <div className="space-y-2">
            <div className="flex justify-between items-center">
              <span className="text-sm text-gray-600">View to Save Rate</span>
              <span className="font-bold text-blue-600">
                {insights.conversion_rates.view_to_save.toFixed(1)}%
              </span>
            </div>
            <div className="flex justify-between items-center">
              <span className="text-sm text-gray-600">Save to Visit Rate</span>
              <span className="font-bold text-purple-600">
                {insights.conversion_rates.save_to_visit.toFixed(1)}%
              </span>
            </div>
          </div>
        </div>

        {/* Buyer Segments */}
        <div>
          <h4 className="font-semibold mb-3">Buyer Types</h4>
          <div className="space-y-2">
            {insights.high_intent_buyers > 0 && (
              <div className="flex items-center justify-between p-2 bg-green-50 rounded">
                <span className="text-sm">Ready to Buy</span>
                <span className="font-bold text-green-600">{insights.high_intent_buyers}</span>
              </div>
            )}
            {insights.active_browsers > 0 && (
              <div className="flex items-center justify-between p-2 bg-blue-50 rounded">
                <span className="text-sm">Actively Searching</span>
                <span className="font-bold text-blue-600">{insights.active_browsers}</span>
              </div>
            )}
            {insights.researchers > 0 && (
              <div className="flex items-center justify-between p-2 bg-yellow-50 rounded">
                <span className="text-sm">Researching</span>
                <span className="font-bold text-yellow-600">{insights.researchers}</span>
              </div>
            )}
          </div>
        </div>

        {/* Engagement Breakdown */}
        <div>
          <h4 className="font-semibold mb-3">Engagement Levels</h4>
          <div className="space-y-2">
            <div className="flex items-center gap-2">
              <div className="flex-1 bg-gray-200 rounded-full h-2">
                <div 
                  className="bg-green-600 h-2 rounded-full" 
                  style={{ width: `${(insights.engagement_breakdown.high / insights.total_engaged_users) * 100}%` }}
                />
              </div>
              <span className="text-sm font-semibold">{insights.engagement_breakdown.high} High</span>
            </div>
            <div className="flex items-center gap-2">
              <div className="flex-1 bg-gray-200 rounded-full h-2">
                <div 
                  className="bg-yellow-600 h-2 rounded-full" 
                  style={{ width: `${(insights.engagement_breakdown.medium / insights.total_engaged_users) * 100}%` }}
                />
              </div>
              <span className="text-sm font-semibold">{insights.engagement_breakdown.medium} Medium</span>
            </div>
            <div className="flex items-center gap-2">
              <div className="flex-1 bg-gray-200 rounded-full h-2">
                <div 
                  className="bg-gray-600 h-2 rounded-full" 
                  style={{ width: `${(insights.engagement_breakdown.low / insights.total_engaged_users) * 100}%` }}
                />
              </div>
              <span className="text-sm font-semibold">{insights.engagement_breakdown.low} Low</span>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};
