import React, { useState } from 'react';
import { usePricingRecommendation } from '../../hooks/pricing/usePricing';
import { TrendingUp, TrendingDown, AlertCircle, CheckCircle } from 'lucide-react';

interface Props {
  recommendation: any;
}

export const PricingRecommendationCard: React.FC<Props> = ({ recommendation }) => {
  const [showRejectDialog, setShowRejectDialog] = useState(false);
  const [rejectReason, setRejectReason] = useState('');
  const [isAccepting, setIsAccepting] = useState(false);
  const [isRejecting, setIsRejecting] = useState(false);

  if (!recommendation) {
    return (
      <div className="bg-white rounded-lg shadow p-6 text-center text-gray-500">
        No pricing recommendation
      </div>
    );
  }

  const isIncrease = recommendation.price_change_percentage > 0;
  const changeColor = isIncrease ? 'text-green-600' : 'text-red-600';
  const TrendIcon = isIncrease ? TrendingUp : TrendingDown;

  const handleAccept = async () => {
    if (window.confirm('Are you sure you want to apply this price change?')) {
      setIsAccepting(true);
      try {
        const token = localStorage.getItem('sellerToken');
        const response = await fetch(`http://localhost:8090/api/pricing/recommendation/${recommendation.id}/accept`, {
          method: 'POST',
          headers: { 'Authorization': `Bearer ${token}` }
        });
        
        const data = await response.json();
        console.log('Accept response:', data);
        
        if (!response.ok || !data.success) {
          alert(`Failed to accept: ${data.detail || data.message || 'Unknown error'}`);
          setIsAccepting(false);
          return;
        }
        
        alert('Price updated successfully!');
        window.location.reload();
      } catch (error) {
        console.error('Accept error:', error);
        alert('Failed to accept recommendation');
        setIsAccepting(false);
      }
    }
  };

  const handleReject = async () => {
    if (rejectReason.trim()) {
      setIsRejecting(true);
      try {
        const token = localStorage.getItem('sellerToken');
        await fetch(`http://localhost:8090/api/pricing/recommendation/${recommendation.id}/reject`, {
          method: 'POST',
          headers: { 'Authorization': `Bearer ${token}`, 'Content-Type': 'application/json' },
          body: JSON.stringify({ reason: rejectReason })
        });
        setShowRejectDialog(false);
        setRejectReason('');
        window.location.reload();
      } catch (error) {
        alert('Failed to reject recommendation');
      } finally {
        setIsRejecting(false);
      }
    }
  };

  const getConfidenceBadge = () => {
    const colors = {
      HIGH: 'bg-green-100 text-green-800',
      MEDIUM: 'bg-yellow-100 text-yellow-800',
      LOW: 'bg-gray-100 text-gray-800'
    };
    return colors[recommendation.confidence_level as keyof typeof colors] || colors.LOW;
  };

  return (
    <>
      <div className="bg-white rounded-lg shadow-lg border-2 border-blue-200">
        <div className="bg-gradient-to-r from-blue-50 to-indigo-50 p-6 rounded-t-lg">
          <div className="flex justify-between items-start">
            <div>
              <h3 className="text-xl font-bold text-gray-800">AI Pricing Recommendation</h3>
              <p className="text-sm text-gray-600 mt-1">{recommendation.title || `Property #${recommendation.property_id}`}</p>
            </div>
            <span className={`px-3 py-1 rounded-full text-sm font-semibold ${getConfidenceBadge()}`}>
              {recommendation.confidence_level} Confidence
            </span>
          </div>
        </div>

        <div className="p-6 space-y-6">
          {/* Price Comparison */}
          <div className="grid grid-cols-3 gap-4 items-center">
            <div className="text-center">
              <p className="text-sm text-gray-600 mb-1">Current Price</p>
              <p className="text-2xl font-bold text-gray-800">
                ₹{recommendation.current_price.toLocaleString('en-IN')}
              </p>
            </div>

            <div className="flex justify-center">
              <TrendIcon className={`w-8 h-8 ${changeColor}`} />
            </div>

            <div className="text-center">
              <p className="text-sm text-gray-600 mb-1">Recommended Price</p>
              <p className={`text-2xl font-bold ${changeColor}`}>
                ₹{recommendation.recommended_price.toLocaleString('en-IN')}
              </p>
              <p className={`text-sm font-semibold ${changeColor}`}>
                {isIncrease ? '+' : ''}{recommendation.price_change_percentage.toFixed(1)}%
              </p>
            </div>
          </div>

          {/* AI Reasoning */}
          <div className="bg-blue-50 p-4 rounded-lg">
            <div className="flex items-start gap-2">
              <AlertCircle className="w-5 h-5 text-blue-600 mt-0.5 flex-shrink-0" />
              <div className="flex-1">
                <h4 className="font-semibold text-gray-800 mb-2">AI Analysis:</h4>
                <p className="text-sm text-gray-700 whitespace-pre-line">
                  {recommendation.reasoning}
                </p>
              </div>
            </div>
          </div>

          {/* ML Model Scores */}
          {(() => {
            const modelScores = recommendation.model_scores;
            if (!modelScores) return null;
            
            const mlValuation = modelScores.ml_valuation;
            const market = modelScores.market;
            
            return (
              <div className="grid grid-cols-2 gap-3">
                {mlValuation && mlValuation.metrics && mlValuation.metrics.estimated_value && (
                  <div className="bg-gradient-to-br from-purple-50 to-indigo-50 p-3 rounded-lg border border-purple-200">
                    <p className="text-xs text-purple-600 font-semibold mb-1">ML Fair Value</p>
                    <p className="text-lg font-bold text-purple-700">
                      ₹{mlValuation.metrics.estimated_value.toLocaleString('en-IN')}
                    </p>
                    <p className="text-xs text-purple-600 mt-1">
                      Confidence: {mlValuation.metrics.confidence}%
                    </p>
                  </div>
                )}
                {market && market.metrics && market.metrics.ml_predicted_price && (
                  <div className="bg-gradient-to-br from-green-50 to-emerald-50 p-3 rounded-lg border border-green-200">
                    <p className="text-xs text-green-600 font-semibold mb-1">ML Market Price</p>
                    <p className="text-lg font-bold text-green-700">
                      ₹{market.metrics.ml_predicted_price.toLocaleString('en-IN')}
                    </p>
                    <p className="text-xs text-green-600 mt-1">
                      Diff: {market.metrics.price_diff_pct}%
                    </p>
                  </div>
                )}
              </div>
            );
          })()}

          {/* Buyer Metrics Summary with PSF Forecast */}
          {(() => {
            const hasStoredMetrics = recommendation.buyer_metrics && 
              (recommendation.buyer_metrics.total_views > 0 || 
               recommendation.buyer_metrics.total_saves > 0 || 
               recommendation.buyer_metrics.total_visits > 0);
            
            let metrics = recommendation.buyer_metrics;
            if (typeof metrics === 'string') {
              try {
                metrics = JSON.parse(metrics);
              } catch (e) {
                metrics = {};
              }
            }
            
            if (!hasStoredMetrics) return null;
            
            return (
              <>
                <div className="grid grid-cols-4 gap-3 text-center">
                  <div className="bg-gray-50 p-3 rounded">
                    <p className="text-2xl font-bold text-blue-600">
                      {metrics.high_intent_buyers || 0}
                    </p>
                    <p className="text-xs text-gray-600">High-Intent Buyers</p>
                  </div>
                  <div className="bg-gray-50 p-3 rounded">
                    <p className="text-2xl font-bold text-green-600">
                      {metrics.total_views || 0}
                    </p>
                    <p className="text-xs text-gray-600">Total Views</p>
                  </div>
                  <div className="bg-gray-50 p-3 rounded">
                    <p className="text-2xl font-bold text-purple-600">
                      {metrics.total_saves || 0}
                    </p>
                    <p className="text-xs text-gray-600">Saves</p>
                  </div>
                  <div className="bg-gray-50 p-3 rounded">
                    <p className="text-2xl font-bold text-orange-600">
                      {metrics.total_visits || 0}
                    </p>
                    <p className="text-xs text-gray-600">Visits</p>
                  </div>
                </div>
                
                {/* PSF Forecast */}
                {metrics.psf_forecast && Object.keys(metrics.psf_forecast).length > 0 && (
                  <div className="bg-gradient-to-r from-amber-50 to-yellow-50 p-4 rounded-lg border border-amber-200">
                    <h4 className="text-sm font-semibold text-amber-800 mb-2">📈 Price Per Sqft Forecast</h4>
                    <div className="flex gap-4 justify-around">
                      {Object.entries(metrics.psf_forecast).map(([year, psf]) => (
                        <div key={year} className="text-center">
                          <p className="text-xs text-amber-600 font-medium">{year}</p>
                          <p className="text-lg font-bold text-amber-700">₹{psf.toLocaleString('en-IN')}</p>
                        </div>
                      ))}
                    </div>
                  </div>
                )}
              </>
            );
          })()}

          {/* Action Buttons */}
          {recommendation.status === 'pending' && (
            <div className="flex gap-3">
              <button
                onClick={handleAccept}
                disabled={isAccepting}
                style={{ backgroundColor: '#2563eb', color: '#ffffff' }}
                className="flex-1 font-semibold py-3 px-4 rounded-lg flex items-center justify-center gap-2 disabled:opacity-50"
              >
                <CheckCircle className="w-4 h-4" style={{ color: '#ffffff' }} />
                {isAccepting ? 'Applying...' : 'Accept & Apply'}
              </button>
              <button
                onClick={() => setShowRejectDialog(true)}
                disabled={isRejecting}
                style={{ backgroundColor: '#ffffff', color: '#374151', border: '2px solid #d1d5db' }}
                className="flex-1 font-semibold py-3 px-4 rounded-lg disabled:opacity-50 hover:bg-gray-50"
              >
                Reject
              </button>
            </div>
          )}
          
          {recommendation.status === 'accepted' && (
            <div className="bg-green-50 border border-green-200 rounded-lg p-4 text-center">
              <p className="text-green-700 font-semibold">Recommendation Accepted & Applied</p>
              <p className="text-sm text-green-600 mt-1">Price updated successfully</p>
            </div>
          )}
          
          {recommendation.status === 'rejected' && (
            <div className="bg-red-50 border border-red-200 rounded-lg p-4 text-center">
              <p className="text-red-700 font-semibold">Recommendation Rejected</p>
              {recommendation.rejection_reason && (
                <p className="text-sm text-red-600 mt-1">Reason: {recommendation.rejection_reason}</p>
              )}
            </div>
          )}

          {/* Expiry Notice */}
          <p className="text-xs text-gray-500 text-center">
            This recommendation expires in {Math.ceil((new Date(recommendation.expires_at).getTime() - Date.now()) / (1000 * 60 * 60 * 24))} days
          </p>
        </div>
      </div>

      {/* Reject Dialog */}
      {showRejectDialog && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 max-w-md w-full mx-4">
            <h3 className="text-lg font-bold mb-4">Reject Recommendation</h3>
            <textarea
              placeholder="Please provide a reason for rejecting this recommendation..."
              value={rejectReason}
              onChange={(e) => setRejectReason(e.target.value)}
              rows={4}
              className="w-full border rounded-lg p-3 mb-4"
            />
            <div className="flex gap-2">
              <button
                onClick={handleReject}
                disabled={!rejectReason.trim()}
                style={{ backgroundColor: '#2563eb', color: '#ffffff' }}
                className="flex-1 font-semibold py-2 px-4 rounded-lg disabled:opacity-50"
              >
                Submit
              </button>
              <button
                onClick={() => setShowRejectDialog(false)}
                style={{ backgroundColor: '#ffffff', color: '#374151', border: '2px solid #d1d5db' }}
                className="flex-1 font-semibold py-2 px-4 rounded-lg hover:bg-gray-50"
              >
                Cancel
              </button>
            </div>
          </div>
        </div>
      )}
    </>
  );
};
