import React, { useState } from 'react';
import { Button } from '@/components/ui/button';
import { agreementsApi } from '@/api/agreements';
import { toast } from 'sonner';

interface InitiatePurchaseButtonProps {
  propertyId: number;
  propertyTitle: string;
  propertyAddress: string;
  salePrice: number;
  sellerId: string;
  sellerName: string;
  sellerEmail: string;
}

export const InitiatePurchaseButton: React.FC<InitiatePurchaseButtonProps> = ({
  propertyId,
  propertyTitle,
  propertyAddress,
  salePrice,
  sellerId,
  sellerName,
  sellerEmail
}) => {
  const [loading, setLoading] = useState(false);

  const handleInitiatePurchase = async () => {
    setLoading(true);
    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      
      const templates = await agreementsApi.getTemplates('purchase');
      
      if (!templates || templates.length === 0) {
        toast.error('Purchase agreement template not found');
        return;
      }

      const content = templates[0].content
        .replace('{buyer_name}', user.full_name || user.email)
        .replace('{buyer_email}', user.email)
        .replace('{seller_name}', sellerName)
        .replace('{property_address}', propertyAddress)
        .replace('{sale_price}', salePrice.toLocaleString());

      const parties = [
        {
          user_id: user.user_id,
          role: 'buyer',
          name: user.full_name || user.email,
          email: user.email
        },
        {
          user_id: sellerId,
          role: 'seller',
          name: sellerName,
          email: sellerEmail
        }
      ];

      await agreementsApi.create({
        template_id: templates[0].id,
        agreement_type: 'purchase',
        property_id: propertyId,
        parties: parties,
        content: content,
        metadata: {
          sale_price: salePrice,
          property_title: propertyTitle
        }
      });

      toast.success('Purchase agreement created! Check Digital Agreements section.');
    } catch (error: any) {
      console.error('Failed to create agreement:', error);
      toast.error('Failed to create agreement: ' + (error.message || 'Unknown error'));
    } finally {
      setLoading(false);
    }
  };

  return (
    <Button
      onClick={handleInitiatePurchase}
      disabled={loading}
      className="w-full bg-blue-600 hover:bg-blue-700 text-white"
      size="lg"
    >
      {loading ? 'Creating Agreement...' : 'Initiate Purchase Agreement'}
    </Button>
  );
};
