import React from 'react';
import { useFormContext } from 'react-hook-form';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '../ui/select';
import { Input } from '../ui/input';
import { Label } from '../ui/label';
import { Card, CardContent, CardHeader, CardTitle } from '../ui/card';

const AREA_UNITS = [
  { value: 'sqft', label: 'Square Feet (sq ft)' },
  { value: 'sqm', label: 'Square Meters (sq m)' },
  { value: 'acres', label: 'Acres' },
  { value: 'hectares', label: 'Hectares' },
  { value: 'bigha', label: 'Bigha' },
  { value: 'katha', label: 'Katha' },
  { value: 'gunta', label: 'Gunta' },
  { value: 'cent', label: 'Cent' }
];

export default function AreaUnitConverter() {
  const { register, watch, setValue } = useFormContext();
  const areaUnit = watch('areaUnit') || 'sqft';
  const area = watch('builtUpArea');

  const convertArea = (value: number, fromUnit: string, toUnit: string): number => {
    // Convert to square feet first, then to target unit
    let sqft = value;
    
    switch (fromUnit) {
      case 'sqm': sqft = value * 10.764; break;
      case 'acres': sqft = value * 43560; break;
      case 'hectares': sqft = value * 107639; break;
      case 'bigha': sqft = value * 27225; break; // Varies by region
      case 'katha': sqft = value * 1361; break; // Varies by region
      case 'gunta': sqft = value * 1089; break;
      case 'cent': sqft = value * 435.6; break;
    }

    switch (toUnit) {
      case 'sqft': return sqft;
      case 'sqm': return sqft / 10.764;
      case 'acres': return sqft / 43560;
      case 'hectares': return sqft / 107639;
      case 'bigha': return sqft / 27225;
      case 'katha': return sqft / 1361;
      case 'gunta': return sqft / 1089;
      case 'cent': return sqft / 435.6;
      default: return sqft;
    }
  };

  return (
    <Card>
      <CardHeader>
        <CardTitle>Area Measurement</CardTitle>
      </CardHeader>
      <CardContent className="space-y-4">
        <div className="grid grid-cols-2 gap-4">
          <div>
            <Label htmlFor="builtUpArea">Built-up Area</Label>
            <Input
              id="builtUpArea"
              type="number"
              placeholder="Enter area"
              {...register('builtUpArea', { valueAsNumber: true })}
            />
          </div>
          <div>
            <Label htmlFor="areaUnit">Unit</Label>
            <Select
              value={areaUnit}
              onValueChange={(value) => setValue('areaUnit', value)}
            >
              <SelectTrigger>
                <SelectValue placeholder="Select unit" />
              </SelectTrigger>
              <SelectContent>
                {AREA_UNITS.map((unit) => (
                  <SelectItem key={unit.value} value={unit.value}>
                    {unit.label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </div>
        
        {area && areaUnit !== 'sqft' && (
          <div className="text-sm text-gray-600 bg-gray-50 p-3 rounded">
            <strong>Conversion:</strong> {area} {areaUnit} = {convertArea(area, areaUnit, 'sqft').toFixed(2)} sq ft
          </div>
        )}
      </CardContent>
    </Card>
  );
}