import { Switch, Route, Router as WouterRouter } from "wouter";
import { Layout } from "./components/layout/layout";
import { queryClient } from "./lib/queryClient";
import { QueryClientProvider } from "@tanstack/react-query";
import { TooltipProvider } from "./components/ui/tooltip";
import { routes } from "./routes";
import { AppProvider } from "./context/AppContext";
import { useEffect } from "react";
import { getCampaignsByUserId } from "../../../api/campaigns";
import { getCurrentUserUuid } from "../../../utils/userUtils";

function Router() {
  return (
    <Switch>
      {routes.map((route) => (
        <Route key={route.path} path={route.path} component={route.component} />
      ))}
    </Switch>
  );
}

function App() {
  useEffect(() => {
    const checkProfileCompletion = async () => {
      const sellerToken = localStorage.getItem('sellerToken');
      const isRoleSwitching = sessionStorage.getItem('original_login_role');
      const justCompleted = sessionStorage.getItem('profileJustCompleted');
      const profileCompleted = localStorage.getItem('sellerProfileCompleted');
      
      // Skip check if role switching, no token, just completed profile, or profile is already marked as completed
      if (isRoleSwitching || !sellerToken || justCompleted || profileCompleted === 'true') {
        if (justCompleted) sessionStorage.removeItem('profileJustCompleted');
        return;
      }
      
      try {
        const response = await fetch('http://localhost:8090/api/seller-profile/completion-status', {
          headers: {
            'Authorization': `Bearer ${sellerToken}`,
            'Content-Type': 'application/json'
          }
        });
        
        if (response.ok) {
          const data = await response.json();
          if (data.completion_percentage < 70) {
            window.location.href = '/seller/profile-completion';
          } else {
            // Mark profile as completed to avoid future checks
            localStorage.setItem('sellerProfileCompleted', 'true');
          }
        }
      } catch (error) {
        console.log('Profile check failed');
      }
    };
    
    checkProfileCompletion();
  }, []);

  useEffect(() => {
    let intervalId: number | undefined;

    const checkApprovedCampaigns = async () => {
      const token =
        localStorage.getItem('sellerToken') ||
        localStorage.getItem('access_token') ||
        localStorage.getItem('agentToken') ||
        localStorage.getItem('token') ||
        localStorage.getItem('auth_token');
      const currentUserId = getCurrentUserUuid();

      if (!token || !currentUserId) return;

      const redirectKey = `campaignApprovedRedirected:${currentUserId}`;
      const alreadyRedirected = sessionStorage.getItem(redirectKey);
      if (alreadyRedirected === 'true') return;

      try {
        const response = await getCampaignsByUserId(currentUserId, { status: 'approved' });
        const approvedCount = (response?.campaigns || []).length;

        if (approvedCount > 0) {
          sessionStorage.setItem(redirectKey, 'true');
          window.location.href = '/seller/dashboard/campaigns';
        }
      } catch (error) {
        // Silent fail to avoid interrupting seller dashboard
      }
    };

    checkApprovedCampaigns();
    intervalId = window.setInterval(checkApprovedCampaigns, 30000);

    return () => {
      if (intervalId) window.clearInterval(intervalId);
    };
  }, []);

  return (
    <QueryClientProvider client={queryClient}>
      <AppProvider>
        <TooltipProvider>
          <WouterRouter base="/seller/dashboard">
            <Layout>
              <Router />
            </Layout>
          </WouterRouter>
        </TooltipProvider>
      </AppProvider>
    </QueryClientProvider>
  );
}

export default App;
