import React, { useState } from "react";
import { toast } from "sonner";
import { Button } from "../components/ui/button";
import { Input } from "../components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "../components/ui/select";
import { Textarea } from "../components/ui/textarea";
import { Label } from "../components/ui/label";
import { api } from "../utils/api";

interface AssetFormData {
  title: string;
  type: string;
  address: string;
  city: string;
  state: string;
  pincode: string;
  area: number;
  price?: number;
  bedrooms?: number;
  bathrooms?: number;
}

interface AddAssetModalProps {
  onClose: () => void;
}

const AddAssetModal: React.FC<AddAssetModalProps> = ({ onClose }) => {
  const [loading, setLoading] = useState(false);
  const [formData, setFormData] = useState<AssetFormData>({
    title: "",
    type: "",
    address: "",
    city: "",
    state: "",
    pincode: "",
    area: 0,
    price: undefined,
    bedrooms: undefined,
    bathrooms: undefined,
  });

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    console.log("=== FORM SUBMIT STARTED ===");
    console.log("Form data:", formData);

    const missingFields = [];
    if (!formData.title.trim()) missingFields.push("Property Title");
    if (!formData.type) missingFields.push("Property Type");
    if (!formData.address.trim()) missingFields.push("Address");
    if (!formData.city.trim()) missingFields.push("City");
    if (!formData.area || formData.area <= 0) missingFields.push("Area");
    if (!formData.pincode.trim()) missingFields.push("Pincode");

    if (missingFields.length > 0) {
      console.log("Validation failed - missing fields:", missingFields);
      toast.error(`Please fill in: ${missingFields.join(", ")}`);
      return;
    }

    console.log("Validation passed, starting API call...");
    setLoading(true);
    try {
      const payload = {
        title: formData.title,
        property_type: formData.type,
        address: formData.address,
        city: formData.city,
        state: formData.state,
        pincode: formData.pincode,
        built_up_area: formData.area,
        bedrooms: formData.bedrooms,
        bathrooms: formData.bathrooms,
        description: "",
        listing_type: "sell", // Assets still have a listing type for future use
        status: "pending", // Will be set by backend based on isAsset flag
        listing_status: "not_listed", // Will be set by backend
        sale_price: formData.price || 0,
        email: "",
        fullname: "",
        property_manager_contact: "",
        isAsset: true, // This is the key flag that makes it an asset
        visibility: "private", // Assets are private by default
      };

      console.log("API payload:", payload);
      const result = await api.createProperty(payload);
      console.log("API result:", result);

      toast.success("Asset added successfully!", {
        description: "Your asset has been submitted and will be available in your portfolio after admin approval.",
      });

      // Trigger property refresh event
      window.dispatchEvent(new CustomEvent("propertyAdded", { detail: true }));

      onClose();
    } catch (error: any) {
      console.error("Error creating asset:", error);
      const errorMessage =
        error.message || "Failed to add asset. Please try again.";
      toast.error(errorMessage);
    } finally {
      setLoading(false);
      console.log("=== FORM SUBMIT ENDED ===");
    }
  };

  const handleInputChange = (
    field: keyof AssetFormData,
    value: string | number,
  ) => {
    console.log(`Updating ${field}:`, value);
    setFormData((prev) => {
      const updated = { ...prev, [field]: value };
      console.log("Updated form data:", updated);
      return updated;
    });
  };

  return (
    <form onSubmit={handleSubmit} className="space-y-4">
      <div>
        <Label htmlFor="title">Property Title *</Label>
        <Input
          id="title"
          value={formData.title}
          onChange={(e) => handleInputChange("title", e.target.value)}
          placeholder="e.g., 3BHK Apartment in Whitefield"
          required
        />
      </div>

      <div>
        <Label htmlFor="type">Property Type *</Label>
        <Select
          value={formData.type}
          onValueChange={(value) => handleInputChange("type", value)}
        >
          <SelectTrigger>
            <SelectValue placeholder="Select property type" />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="apartment">Apartment</SelectItem>
            <SelectItem value="house">House</SelectItem>
            <SelectItem value="villa">Villa</SelectItem>
            <SelectItem value="commercial">Commercial</SelectItem>
          </SelectContent>
        </Select>
      </div>

      <div>
        <Label htmlFor="address">Address *</Label>
        <Input
          id="address"
          value={formData.address}
          onChange={(e) => handleInputChange("address", e.target.value)}
          placeholder="Enter full address"
          required
        />
      </div>

      <div className="grid grid-cols-2 gap-4">
        <div>
          <Label htmlFor="city">City *</Label>
          <Input
            id="city"
            value={formData.city}
            onChange={(e) => handleInputChange("city", e.target.value)}
            placeholder="e.g., Bangalore"
            required
          />
        </div>
        <div>
          <Label htmlFor="state">State *</Label>
          <Input
            id="state"
            value={formData.state}
            onChange={(e) => handleInputChange("state", e.target.value)}
            placeholder="e.g., Karnataka"
            required
          />
        </div>
      </div>

      <div>
        <Label htmlFor="pincode">Pincode *</Label>
        <Input
          id="pincode"
          value={formData.pincode}
          onChange={(e) => handleInputChange("pincode", e.target.value)}
          placeholder="e.g., 560001"
          required
        />
      </div>

      <div className="grid grid-cols-2 gap-4">
        <div>
          <Label htmlFor="area">Built-up Area (sq ft) *</Label>
          <Input
            id="area"
            type="number"
            value={formData.area || ""}
            onChange={(e) =>
              handleInputChange("area", parseInt(e.target.value) || 0)
            }
            placeholder="e.g., 1200"
            required
          />
        </div>
        <div>
          <Label htmlFor="price">Property Value (₹)</Label>
          <Input
            id="price"
            type="number"
            value={formData.price || ""}
            onChange={(e) =>
              handleInputChange("price", parseInt(e.target.value) || undefined)
            }
            placeholder="e.g., 5000000 (optional)"
          />
        </div>
      </div>

      <div className="grid grid-cols-2 gap-4">
        <div>
          <Label htmlFor="bedrooms">Bedrooms</Label>
          <Input
            id="bedrooms"
            type="number"
            value={formData.bedrooms || ""}
            onChange={(e) =>
              handleInputChange(
                "bedrooms",
                parseInt(e.target.value) || undefined,
              )
            }
            placeholder="e.g., 3"
          />
        </div>
        <div>
          <Label htmlFor="bathrooms">Bathrooms</Label>
          <Input
            id="bathrooms"
            type="number"
            value={formData.bathrooms || ""}
            onChange={(e) =>
              handleInputChange(
                "bathrooms",
                parseInt(e.target.value) || undefined,
              )
            }
            placeholder="e.g., 2"
          />
        </div>
      </div>

      <div className="flex gap-4 pt-4">
        <Button
          type="button"
          variant="outline"
          onClick={onClose}
          className="flex-1"
        >
          Cancel
        </Button>
        <Button
          type="submit"
          disabled={loading}
          className="flex-1 bg-[#0056D2] hover:bg-blue-700"
          onClick={(e) => {
            console.log("Add to Assets button clicked!");
            handleSubmit(e);
          }}
        >
          {loading ? "Adding Asset..." : "Add to Assets"}
        </Button>
      </div>
    </form>
  );
};

export default AddAssetModal;
