import React from 'react';
import { Property } from '../types';
import { MapPin, Bed, Bath, Square, Calendar, Eye, Edit, Trash2, Heart, Share2 } from 'lucide-react';

interface AgentStylePropertyCardProps {
  property: Property;
  onView: (id: string) => void;
  onEdit: (id: string) => void;
  onDelete: (id: string) => void;
}

const AgentStylePropertyCard: React.FC<AgentStylePropertyCardProps> = ({
  property,
  onView,
  onEdit,
  onDelete
}) => {
  const getStatusBadge = (listingIntent: string) => {
    switch (listingIntent) {
      case 'urgent-sale': return 'bg-red-500 text-white';
      case 'rent': return 'bg-blue-500 text-white';
      default: return 'bg-green-500 text-white';
    }
  };

  const getStatusText = (listingIntent: string) => {
    switch (listingIntent) {
      case 'urgent-sale': return 'Urgent Sale';
      case 'rent': return 'For Rent';
      default: return 'For Sale';
    }
  };

  return (
    <div className="group bg-white rounded-lg border border-gray-200 hover:border-gray-300 hover:shadow-xl transition-all duration-300 overflow-hidden">
      {/* Image Section */}
      <div className="relative h-48 overflow-hidden">
        <img
          src={
            (property.images && property.images.length > 0) 
              ? (typeof property.images[0] === 'string' ? property.images[0] : property.images[0]?.url || property.images[0]?.image)
              : "https://images.unsplash.com/photo-1600596542815-ffad4c1539a9?ixlib=rb-4.0.3&auto=format&fit=crop&w=800&h=400"
          }
          alt={property.title}
          className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-500"
          onError={(e) => {
            e.currentTarget.src = "https://images.unsplash.com/photo-1600596542815-ffad4c1539a9?ixlib=rb-4.0.3&auto=format&fit=crop&w=800&h=400";
          }}
        />
        
        {/* Status Badge */}
        <div className={`absolute top-3 left-3 px-2 py-1 rounded text-xs font-semibold ${getStatusBadge(property.listingIntent)}`}>
          {getStatusText(property.listingIntent)}
        </div>
        
        {/* Quick Actions */}
        <div className="absolute top-3 right-3 flex gap-1 opacity-0 group-hover:opacity-100 transition-opacity duration-300">
          <button className="w-8 h-8 bg-white/90 hover:bg-white rounded-full flex items-center justify-center shadow-sm">
            <Heart className="w-4 h-4 text-gray-600" />
          </button>
          <button className="w-8 h-8 bg-white/90 hover:bg-white rounded-full flex items-center justify-center shadow-sm">
            <Share2 className="w-4 h-4 text-gray-600" />
          </button>
        </div>
        

      </div>

      {/* Content Section */}
      <div className="p-4">
        {/* Title & Location */}
        <div className="mb-3">
          <h3 className="text-lg font-semibold text-gray-900 mb-1 line-clamp-1">
            {property.title}
          </h3>
          <div className="flex items-center text-gray-600 text-sm mb-2">
            <MapPin className="w-4 h-4 mr-1" />
            <span className="line-clamp-1">{property.city}, {property.state}</span>
          </div>
          <div className="text-xl font-bold text-gray-900">
            ₹{(property.price / 100000).toFixed(1)}L
            {property.listingIntent === 'rent' && <span className="text-sm font-normal text-gray-600"> /mo</span>}
          </div>
        </div>

        {/* Property Features */}
        <div className="flex items-center justify-between mb-4 py-2 px-3 bg-gray-50 rounded">
          {property.bhk && (
            <div className="flex items-center gap-1 text-sm text-gray-700">
              <Bed className="w-4 h-4" />
              <span>{property.bhk}</span>
            </div>
          )}
          <div className="flex items-center gap-1 text-sm text-gray-700">
            <Bath className="w-4 h-4" />
            <span>2</span>
          </div>
          <div className="flex items-center gap-1 text-sm text-gray-700">
            <Square className="w-4 h-4" />
            <span>{property.area} sqft</span>
          </div>
          <div className="flex items-center gap-1 text-sm text-gray-500">
            <Calendar className="w-4 h-4" />
            <span>{new Date(property.createdAt).getFullYear()}</span>
          </div>
        </div>

        {/* Action Buttons */}
        <div className="flex gap-2">
          <button 
            onClick={() => onView(property.id)}
            className="flex-1 bg-black hover:bg-gray-800 text-white py-2 px-3 rounded text-sm font-medium transition-colors flex items-center justify-center gap-1"
          >
            <Eye className="w-4 h-4" />
            View Details
          </button>
          <button 
            onClick={() => onDelete(property.id)}
            className="px-3 py-2 border border-red-200 hover:border-red-300 hover:bg-red-50 text-red-600 rounded text-sm font-medium transition-colors"
          >
            <Trash2 className="w-4 h-4" />
          </button>
        </div>

        {/* Footer Stats */}
        <div className="mt-3 pt-3 border-t border-gray-100 flex items-center justify-between text-xs text-gray-500">
          <span>{property.views} views</span>
          <span>Listed {new Date(property.createdAt).toLocaleDateString()}</span>
        </div>
      </div>
    </div>
  );
};

export default AgentStylePropertyCard;