import React, { useState, useEffect } from 'react';
import { toast } from 'sonner';

interface Enquiry {
  id: string;
  user_id: string;
  property_id: number;
  data_content: {
    enquiry_for: 'SALE' | 'RENTAL';
    tenant_name?: string;
    tenant_email?: string;
    tenant_phone?: string;
    preferred_move_in?: string;
    rental_duration?: string;
    message?: string;
  };
  status: 'NEW' | 'CONTACTED' | 'PROSPECT' | 'CONVERTED' | 'ARCHIVED';
  created_at: string;
  property_title: string;
  address: string;
}

export function EnquiryManagement() {
  const [enquiries, setEnquiries] = useState<Enquiry[]>([]);
  const [loading, setLoading] = useState(true);
  const [filter, setFilter] = useState<'ALL' | 'SALE' | 'RENTAL'>('ALL');
  const [statusFilter, setStatusFilter] = useState<string>('ALL');

  useEffect(() => {
    fetchEnquiries();
  }, [filter, statusFilter]);

  const fetchEnquiries = async () => {
    try {
      setLoading(true);
      const params = new URLSearchParams();
      if (filter !== 'ALL') params.append('enquiry_type', filter);
      if (statusFilter !== 'ALL') params.append('status_filter', statusFilter);
      
      const response = await fetch(`/api/property-management/seller/enquiries?${params}`, {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        }
      });
      
      if (response.ok) {
        const data = await response.json();
        setEnquiries(data.enquiries);
      }
    } catch (error) {
      console.error('Failed to fetch enquiries:', error);
      toast.error('Failed to load enquiries');
    } finally {
      setLoading(false);
    }
  };

  const handleStatusUpdate = async (enquiryId: string, newStatus: string) => {
    try {
      const response = await fetch(`/api/property-management/enquiries/${enquiryId}/status`, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        },
        body: JSON.stringify({ status: newStatus })
      });
      
      if (response.ok) {
        toast.success('Status updated successfully');
        fetchEnquiries();
      } else {
        const error = await response.json();
        toast.error(error.detail || 'Failed to update status');
      }
    } catch (error) {
      console.error('Status update failed:', error);
      toast.error('Failed to update status');
    }
  };

  const handleConvertEnquiry = async (enquiryId: string) => {
    try {
      const response = await fetch(`/api/property-management/enquiries/${enquiryId}/convert`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        },
        body: JSON.stringify({
          tenancy_type: 'RENT',
          start_date: new Date().toISOString().split('T')[0]
        })
      });
      
      if (response.ok) {
        toast.success('Enquiry converted to tenancy successfully!');
        fetchEnquiries();
      } else {
        const error = await response.json();
        toast.error(error.detail || 'Failed to convert enquiry');
      }
    } catch (error) {
      console.error('Conversion failed:', error);
      toast.error('Failed to convert enquiry');
    }
  };

  const getStatusOptions = (enquiry: Enquiry) => {
    const isRental = enquiry.data_content.enquiry_for === 'RENTAL';
    const options = ['NEW', 'CONTACTED'];
    
    if (isRental) {
      options.push('CONVERTED');
    } else {
      options.push('PROSPECT');
    }
    
    options.push('ARCHIVED');
    return options;
  };

  const getEnquiryCount = (type: string) => {
    if (type === 'ALL') return enquiries.length;
    return enquiries.filter(e => e.data_content.enquiry_for === type).length;
  };

  if (loading) {
    return <div className="p-6">Loading enquiries...</div>;
  }

  return (
    <div className="p-6">
      <h2 className="text-2xl font-bold mb-6">Leads & Enquiries</h2>
      
      {/* Filter Tabs */}
      <div className="flex gap-4 mb-6">
        <div className="flex bg-gray-100 rounded-lg p-1">
          <button 
            className={`px-4 py-2 rounded transition-colors ${
              filter === 'ALL' ? 'bg-white shadow text-blue-600' : 'text-gray-600 hover:text-gray-800'
            }`}
            onClick={() => setFilter('ALL')}
          >
            All ({getEnquiryCount('ALL')})
          </button>
          <button 
            className={`px-4 py-2 rounded transition-colors ${
              filter === 'SALE' ? 'bg-white shadow text-blue-600' : 'text-gray-600 hover:text-gray-800'
            }`}
            onClick={() => setFilter('SALE')}
          >
            Sale ({getEnquiryCount('SALE')})
          </button>
          <button 
            className={`px-4 py-2 rounded transition-colors ${
              filter === 'RENTAL' ? 'bg-white shadow text-green-600' : 'text-gray-600 hover:text-gray-800'
            }`}
            onClick={() => setFilter('RENTAL')}
          >
            Rental ({getEnquiryCount('RENTAL')})
          </button>
        </div>
        
        <select 
          value={statusFilter} 
          onChange={(e) => setStatusFilter(e.target.value)}
          className="border rounded-lg px-3 py-2 bg-white"
        >
          <option value="ALL">All Status</option>
          <option value="NEW">New</option>
          <option value="CONTACTED">Contacted</option>
          <option value="PROSPECT">Prospects</option>
          <option value="CONVERTED">Converted</option>
          <option value="ARCHIVED">Archived</option>
        </select>
      </div>

      {/* Enquiries List */}
      <div className="space-y-4">
        {enquiries.length === 0 ? (
          <div className="text-center py-8 text-gray-500">
            No enquiries found
          </div>
        ) : (
          enquiries.map((enquiry) => (
            <EnquiryCard 
              key={enquiry.id} 
              enquiry={enquiry} 
              onStatusUpdate={handleStatusUpdate}
              onConvert={handleConvertEnquiry}
              statusOptions={getStatusOptions(enquiry)}
            />
          ))
        )}
      </div>
    </div>
  );
}

interface EnquiryCardProps {
  enquiry: Enquiry;
  onStatusUpdate: (id: string, status: string) => void;
  onConvert: (id: string) => void;
  statusOptions: string[];
}

function EnquiryCard({ enquiry, onStatusUpdate, onConvert, statusOptions }: EnquiryCardProps) {
  const isRental = enquiry.data_content.enquiry_for === 'RENTAL';
  const canConvert = isRental && enquiry.status === 'CONTACTED';

  return (
    <div className="border rounded-lg p-6 bg-white shadow-sm hover:shadow-md transition-shadow">
      <div className="flex justify-between items-start">
        <div className="flex-1">
          {/* Enquiry Type Badge */}
          <span className={`inline-block px-3 py-1 rounded-full text-sm font-medium mb-3 ${
            isRental 
              ? 'bg-green-100 text-green-800' 
              : 'bg-blue-100 text-blue-800'
          }`}>
            {enquiry.data_content.enquiry_for}
          </span>
          
          {/* Customer Info */}
          <div className="mb-4">
            <h3 className="font-semibold text-lg text-gray-900">
              {enquiry.data_content.tenant_name || 'Interested Buyer'}
            </h3>
            <p className="text-gray-600">{enquiry.data_content.tenant_email}</p>
            {enquiry.data_content.tenant_phone && (
              <p className="text-gray-600">{enquiry.data_content.tenant_phone}</p>
            )}
          </div>
          
          {/* Property Info */}
          <div className="mb-4">
            <h4 className="font-medium text-gray-900">{enquiry.property_title}</h4>
            <p className="text-sm text-gray-500">{enquiry.address}</p>
          </div>
          
          {/* Rental Specific Info */}
          {isRental && (
            <div className="mb-4 p-3 bg-green-50 rounded-lg">
              <div className="grid grid-cols-2 gap-4 text-sm">
                {enquiry.data_content.preferred_move_in && (
                  <div>
                    <span className="font-medium text-gray-700">Move-in Date:</span>
                    <p className="text-gray-600">{enquiry.data_content.preferred_move_in}</p>
                  </div>
                )}
                {enquiry.data_content.rental_duration && (
                  <div>
                    <span className="font-medium text-gray-700">Duration:</span>
                    <p className="text-gray-600">{enquiry.data_content.rental_duration}</p>
                  </div>
                )}
              </div>
            </div>
          )}
          
          {/* Message */}
          {enquiry.data_content.message && (
            <div className="mb-4">
              <span className="font-medium text-gray-700">Message:</span>
              <p className="text-gray-600 mt-1">{enquiry.data_content.message}</p>
            </div>
          )}
          
          {/* Timestamp */}
          <p className="text-xs text-gray-400">
            Received: {new Date(enquiry.created_at).toLocaleDateString()}
          </p>
          
          {/* Contact Buttons */}
          {enquiry.data_content.tenant_phone && (
            <div className="flex gap-2 mt-4">
              <button
                onClick={() => {
                  const phone = enquiry.data_content.tenant_phone;
                  if (phone) {
                    const cleanPhone = phone.replace(/[^+\d]/g, "");
                    window.open(`https://wa.me/${cleanPhone}`, "_blank");
                  }
                }}
                className="flex items-center gap-2 px-4 py-2 bg-green-500 hover:bg-green-600 text-white rounded-lg transition-colors text-sm font-medium"
              >
                <svg className="w-4 h-4" viewBox="0 0 24 24" fill="currentColor">
                  <path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893A11.821 11.821 0 0020.465 3.488"/>
                </svg>
                WhatsApp
              </button>
              <button
                onClick={() => {
                  const phone = enquiry.data_content.tenant_phone;
                  if (phone) {
                    window.open(`tel:${phone}`, "_self");
                  }
                }}
                className="flex items-center gap-2 px-4 py-2 bg-blue-500 hover:bg-blue-600 text-white rounded-lg transition-colors text-sm font-medium"
              >
                <svg className="w-4 h-4" viewBox="0 0 24 24" fill="currentColor">
                  <path d="M6.62 10.79c1.44 2.83 3.76 5.14 6.59 6.59l2.2-2.2c.27-.27.67-.36 1.02-.24 1.12.37 2.33.57 3.57.57.55 0 1 .45 1 1V20c0 .55-.45 1-1 1-9.39 0-17-7.61-17-17 0-.55.45-1 1-1h3.5c.55 0 1 .45 1 1 0 1.25.2 2.45.57 3.57.11.35.03.74-.25 1.02l-2.2 2.2z"/>
                </svg>
                Call
              </button>
            </div>
          )}
        </div>
        
        {/* Actions */}
        <div className="flex flex-col gap-3 ml-6">
          {/* Status Dropdown */}
          <select 
            value={enquiry.status}
            onChange={(e) => onStatusUpdate(enquiry.id, e.target.value)}
            className="border rounded-lg px-3 py-2 text-sm bg-white min-w-[120px]"
          >
            {statusOptions.map(status => (
              <option key={status} value={status}>
                {status.charAt(0) + status.slice(1).toLowerCase()}
              </option>
            ))}
          </select>
          
          {/* Convert Button for Rental */}
          {canConvert && (
            <button
              onClick={() => onConvert(enquiry.id)}
              className="bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 transition-colors text-sm font-medium"
            >
              Convert to Tenancy
            </button>
          )}
        </div>
      </div>
    </div>
  );
}