import React from 'react';
import { Property } from '../types';

interface MobilePropertyListProps {
  properties: Property[];
  onEdit: (id: string) => void;
  onDelete: (id: string) => void;
  onView: (id: string) => void;
}

const MobilePropertyList: React.FC<MobilePropertyListProps> = ({
  properties,
  onEdit,
  onDelete,
  onView
}) => {
  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active': return 'bg-primary/80 text-white';
      case 'inactive': return 'bg-gray-500/80 text-white';
      case 'under_review': return 'bg-amber-500/80 text-white';
      default: return 'bg-primary/80 text-white';
    }
  };

  const getStatusLabel = (status: string) => {
    switch (status) {
      case 'active': return 'For Sale';
      case 'inactive': return 'Inactive';
      case 'under_review': return 'Under Review';
      default: return 'For Sale';
    }
  };

  return (
    <div className="max-w-md mx-auto bg-background-light dark:bg-background-dark min-h-screen">
      {/* Header */}
      <header className="sticky top-0 z-10 bg-background-light/80 dark:bg-background-dark/80 backdrop-blur-sm">
        <div className="flex items-center justify-between p-4">
          <button className="p-2 -ml-2">
            <span className="material-symbols-outlined text-zinc-900 dark:text-zinc-100">menu</span>
          </button>
          <h1 className="text-lg font-bold text-zinc-900 dark:text-zinc-100">My Listings</h1>
          <button className="p-2 -mr-2">
            <span className="material-symbols-outlined text-zinc-900 dark:text-zinc-100">add_circle_outline</span>
          </button>
        </div>
      </header>

      {/* Properties List */}
      <main className="p-4 space-y-6 pb-24">
        {properties.map((property) => (
          <div key={property.id} className="bg-white dark:bg-zinc-800 rounded-xl shadow-lg shadow-zinc-200/50 dark:shadow-zinc-900/50 overflow-hidden">
            <div className="relative">
              <img
                alt={property.title}
                className="w-full h-36 object-cover"
                src={property.images?.[0] || "https://images.unsplash.com/photo-1586023492125-27b2c045efd7?ixlib=rb-4.0.3&auto=format&fit=crop&w=400&h=200"}
              />
              <div className={`absolute top-2 left-2 text-xs font-semibold px-2 py-1 rounded-full ${getStatusColor(property.status)}`}>
                {getStatusLabel(property.status)}
              </div>
            </div>
            
            <div className="p-4">
              <h2 className="text-lg font-bold text-zinc-900 dark:text-white">{property.title}</h2>
              <p className="text-sm text-zinc-500 dark:text-zinc-400">{property.city}, {property.state}</p>
              
              <div className="mt-3 flex items-center justify-between">
                <p className="text-lg font-bold text-primary">
                  ₹{(property.price / 100000).toFixed(1)}L
                  {property.listingIntent === 'rent' && ' / month'}
                </p>
              </div>
              
              <div className="mt-4 border-t border-zinc-200 dark:border-zinc-700 pt-4 flex items-center space-x-2 text-zinc-600 dark:text-zinc-300">
                {property.bhk && (
                  <>
                    <div className="flex items-center space-x-1">
                      <span className="material-symbols-outlined text-base">king_bed</span>
                      <span className="text-sm font-medium">{property.bhk}</span>
                    </div>
                    <span className="text-zinc-300 dark:text-zinc-600">|</span>
                  </>
                )}
                <div className="flex items-center space-x-1">
                  <span className="material-symbols-outlined text-base">square_foot</span>
                  <span className="text-sm font-medium">{property.area} sq ft</span>
                </div>
                <span className="text-zinc-300 dark:text-zinc-600">|</span>
                <div className="flex items-center space-x-1">
                  <span className="material-symbols-outlined text-base">chair</span>
                  <span className="text-sm font-medium">{property.type}</span>
                </div>
              </div>
              
              <div className="mt-4 flex gap-2">
                <button 
                  onClick={() => onView(property.id)}
                  className="flex-1 bg-primary text-white py-2.5 rounded-lg text-sm font-semibold hover:bg-primary/90 transition-colors flex items-center justify-center gap-2"
                >
                  <span className="material-symbols-outlined text-base">visibility</span>
                  View Details
                </button>
                <button 
                  onClick={() => onEdit(property.id)}
                  className="p-2.5 bg-blue-50 text-blue-600 rounded-lg hover:bg-blue-100 transition-colors"
                  title="Edit"
                >
                  <span className="material-symbols-outlined text-base">edit</span>
                </button>
                <button 
                  onClick={() => onDelete(property.id)}
                  className="p-2.5 bg-red-50 text-red-600 rounded-lg hover:bg-red-100 transition-colors"
                  title="Delete"
                >
                  <span className="material-symbols-outlined text-base">delete</span>
                </button>
              </div>
            </div>
          </div>
        ))}
      </main>

      {/* Bottom Navigation */}
      <footer className="fixed bottom-0 left-0 right-0 max-w-md mx-auto bg-background-light/80 dark:bg-background-dark/80 backdrop-blur-sm border-t border-zinc-200 dark:border-zinc-800">
        <nav className="flex justify-around items-center h-16">
          <a className="flex flex-col items-center text-zinc-500 dark:text-zinc-400 hover:text-primary dark:hover:text-primary transition-colors" href="#">
            <span className="material-symbols-outlined">home</span>
            <span className="text-xs font-medium">Home</span>
          </a>
          <a className="flex flex-col items-center text-primary dark:text-primary transition-colors" href="#">
            <span className="material-symbols-outlined" style={{fontVariationSettings: "'FILL' 1"}}>list_alt</span>
            <span className="text-xs font-bold">Listings</span>
          </a>
          <a className="flex flex-col items-center text-zinc-500 dark:text-zinc-400 hover:text-primary dark:hover:text-primary transition-colors" href="#">
            <span className="material-symbols-outlined">analytics</span>
            <span className="text-xs font-medium">Analytics</span>
          </a>
          <a className="flex flex-col items-center text-zinc-500 dark:text-zinc-400 hover:text-primary dark:hover:text-primary transition-colors" href="#">
            <span className="material-symbols-outlined">mail</span>
            <span className="text-xs font-medium">Messages</span>
          </a>
          <a className="flex flex-col items-center text-zinc-500 dark:text-zinc-400 hover:text-primary dark:hover:text-primary transition-colors" href="#">
            <span className="material-symbols-outlined">person</span>
            <span className="text-xs font-medium">Profile</span>
          </a>
        </nav>
      </footer>
    </div>
  );
};

export default MobilePropertyList;