import React from 'react';
import { Property } from '../types';

interface PremiumMobileCardProps {
  property: Property;
  onView: (id: string) => void;
  onEdit: (id: string) => void;
  onDelete: (id: string) => void;
}

const PremiumMobileCard: React.FC<PremiumMobileCardProps> = ({
  property,
  onView,
  onEdit,
  onDelete
}) => {
  const getStatusColor = (listingIntent: string) => {
    if (listingIntent === 'urgent-sale') return 'bg-red-500';
    if (listingIntent === 'rent') return 'bg-blue-500';
    return 'bg-green-500';
  };

  const getStatusText = (listingIntent: string) => {
    if (listingIntent === 'urgent-sale') return 'Urgent Sale';
    if (listingIntent === 'rent') return 'For Rent';
    return 'For Sale';
  };

  return (
    <div className="bg-white rounded-2xl shadow-lg shadow-black/5 overflow-hidden border border-gray-100 mx-4 mb-6">
      {/* Image Section */}
      <div className="relative">
        <div className="aspect-[16/10] bg-gray-100">
          <img
            src={property.images?.[0] || "https://images.unsplash.com/photo-1600596542815-ffad4c1539a9?ixlib=rb-4.0.3&auto=format&fit=crop&w=600&h=400"}
            alt={property.title}
            className="w-full h-full object-cover"
          />
        </div>
        
        {/* Status Badge */}
        <div className={`absolute top-3 left-3 ${getStatusColor(property.listingIntent)} text-white px-3 py-1 rounded-full text-xs font-semibold`}>
          {getStatusText(property.listingIntent)}
        </div>
        
        {/* Price Badge */}
        <div className="absolute bottom-3 left-3 bg-white/95 backdrop-blur-sm rounded-lg px-3 py-2">
          <div className="text-lg font-bold text-gray-900">
            ₹{(property.price / 100000).toFixed(1)}L
            {property.listingIntent === 'rent' && <span className="text-xs text-gray-600"> /mo</span>}
          </div>
        </div>
      </div>

      {/* Content Section */}
      <div className="p-4">
        <h3 className="text-lg font-bold text-gray-900 mb-2 line-clamp-2">
          {property.title}
        </h3>
        
        <div className="flex items-center text-gray-600 mb-3">
          <span className="material-symbols-outlined text-base mr-1">location_on</span>
          <span className="text-sm line-clamp-1">{property.city}, {property.state}</span>
        </div>

        {/* Property Features */}
        <div className="flex items-center gap-4 mb-4 text-gray-700">
          {property.bhk && (
            <div className="flex items-center gap-1">
              <span className="material-symbols-outlined text-base">bed</span>
              <span className="text-sm font-medium">{property.bhk}</span>
            </div>
          )}
          <div className="flex items-center gap-1">
            <span className="material-symbols-outlined text-base">bathtub</span>
            <span className="text-sm font-medium">2</span>
          </div>
          <div className="flex items-center gap-1">
            <span className="material-symbols-outlined text-base">square_foot</span>
            <span className="text-sm font-medium">{property.area} sqft</span>
          </div>
        </div>

        {/* Action Buttons */}
        <div className="flex gap-2">
          <button 
            onClick={() => onView(property.id)}
            className="flex-1 bg-blue-600 text-white py-3 rounded-xl font-semibold text-sm hover:bg-blue-700 transition-colors flex items-center justify-center gap-2"
          >
            <span className="material-symbols-outlined text-base">visibility</span>
            View Details
          </button>
          <button 
            onClick={() => onEdit(property.id)}
            className="w-12 h-12 bg-gray-100 text-gray-600 rounded-xl hover:bg-gray-200 transition-colors flex items-center justify-center"
          >
            <span className="material-symbols-outlined text-base">edit</span>
          </button>
          <button 
            onClick={() => onDelete(property.id)}
            className="w-12 h-12 bg-red-50 text-red-600 rounded-xl hover:bg-red-100 transition-colors flex items-center justify-center"
          >
            <span className="material-symbols-outlined text-base">delete</span>
          </button>
        </div>

        {/* Footer Info */}
        <div className="mt-3 pt-3 border-t border-gray-100 text-xs text-gray-500 text-center">
          Listed on {new Date(property.createdAt).toLocaleDateString('en-GB')}
        </div>
      </div>
    </div>
  );
};

export default PremiumMobileCard;