import React from 'react';
import { Property } from '../types';
import { MapPin, Bed, Bath, Square, Heart, Share2, Eye } from 'lucide-react';

interface PremiumPropertyCardProps {
  property: Property;
  onView: (id: string) => void;
  onEdit?: (id: string) => void;
  onDelete?: (id: string) => void;
}

const PremiumPropertyCard: React.FC<PremiumPropertyCardProps> = ({
  property,
  onView,
  onEdit,
  onDelete
}) => {
  const getStatusBadge = (status: string, listingIntent: string) => {
    if (listingIntent === 'urgent-sale') return 'bg-red-500 text-white';
    if (listingIntent === 'rent') return 'bg-blue-500 text-white';
    return 'bg-green-500 text-white';
  };

  const getStatusText = (status: string, listingIntent: string) => {
    if (listingIntent === 'urgent-sale') return 'Urgent Sale';
    if (listingIntent === 'rent') return 'For Rent';
    return 'For Sale';
  };

  return (
    <div className="group bg-white rounded-xl shadow-md hover:shadow-xl transition-all duration-300 overflow-hidden border border-gray-200 hover:border-blue-300">
      {/* Image Section */}
      <div className="relative overflow-hidden">
        <div className="aspect-[5/3] bg-gray-100">
          <img
            src={property.images?.[0] || "https://images.unsplash.com/photo-1600596542815-ffad4c1539a9?ixlib=rb-4.0.3&auto=format&fit=crop&w=800&h=600"}
            alt={property.title}
            className="w-full h-full object-cover"
          />
        </div>
        
        {/* Status Badge */}
        <div className={`absolute top-4 left-4 px-3 py-1.5 rounded-full text-xs font-semibold ${getStatusBadge(property.status, property.listingIntent)}`}>
          {getStatusText(property.status, property.listingIntent)}
        </div>
        
        {/* Action Buttons */}
        <div className="absolute top-4 right-4 flex gap-2 opacity-0 group-hover:opacity-100 transition-opacity duration-300">
          <button className="w-8 h-8 bg-white/90 backdrop-blur-sm rounded-full flex items-center justify-center hover:bg-white transition-colors">
            <Heart className="w-4 h-4 text-gray-600" />
          </button>
          <button className="w-8 h-8 bg-white/90 backdrop-blur-sm rounded-full flex items-center justify-center hover:bg-white transition-colors">
            <Share2 className="w-4 h-4 text-gray-600" />
          </button>
        </div>
        
        {/* Price Overlay */}
        <div className="absolute bottom-4 left-4 bg-white/95 backdrop-blur-sm rounded-lg px-3 py-2">
          <div className="text-2xl font-bold text-gray-900">
            ₹{(property.price / 100000).toFixed(1)}L
            {property.listingIntent === 'rent' && <span className="text-sm font-normal text-gray-600"> /month</span>}
          </div>
        </div>
      </div>

      {/* Content Section */}
      <div className="p-5">
        {/* Title and Location */}
        <div className="mb-4">
          <h3 className="text-lg font-bold text-gray-900 mb-2 line-clamp-2 leading-snug">
            {property.title}
          </h3>
          <div className="flex items-center text-gray-600 mb-3">
            <MapPin className="w-4 h-4 mr-1.5 flex-shrink-0" />
            <span className="text-sm line-clamp-1">{property.city}, {property.state}</span>
          </div>
          <div className="text-2xl font-bold text-blue-600">
            ₹{(property.price / 100000).toFixed(1)}L
            {property.listingIntent === 'rent' && <span className="text-sm font-normal text-gray-600"> /month</span>}
          </div>
        </div>

        {/* Property Features */}
        <div className="flex items-center justify-between mb-4 py-3 px-4 bg-gray-50 rounded-lg">
          {property.bhk && (
            <div className="flex items-center gap-1.5">
              <Bed className="w-4 h-4 text-gray-600" />
              <span className="text-sm font-medium text-gray-700">{property.bhk}</span>
            </div>
          )}
          <div className="flex items-center gap-1.5">
            <Bath className="w-4 h-4 text-gray-600" />
            <span className="text-sm font-medium text-gray-700">2 Bath</span>
          </div>
          <div className="flex items-center gap-1.5">
            <Square className="w-4 h-4 text-gray-600" />
            <span className="text-sm font-medium text-gray-700">{property.area} sqft</span>
          </div>
        </div>

        {/* Action Buttons */}
        <div className="flex gap-2">
          <button 
            onClick={() => onView(property.id)}
            className="flex-1 bg-blue-600 text-white py-2.5 px-4 rounded-lg text-sm font-semibold hover:bg-blue-700 transition-colors flex items-center justify-center gap-2"
          >
            <Eye className="w-4 h-4" />
            View Full Details
          </button>
          {onEdit && (
            <button 
              onClick={() => onEdit(property.id)}
              className="px-4 py-2.5 border border-gray-300 text-gray-700 text-sm font-semibold rounded-lg hover:bg-gray-50 hover:border-gray-400 transition-colors"
            >
              Edit
            </button>
          )}
          {onDelete && (
            <button 
              onClick={() => onDelete(property.id)}
              className="px-4 py-2.5 border border-red-200 text-red-600 text-sm font-semibold rounded-lg hover:bg-red-50 hover:border-red-300 transition-colors"
            >
              Delete
            </button>
          )}
        </div>

        {/* Footer Info */}
        <div className="mt-4 pt-3 border-t border-gray-100 text-xs text-gray-500 text-center">
          Listed on {new Date(property.createdAt).toLocaleDateString('en-GB')}
        </div>
      </div>
    </div>
  );
};

export default PremiumPropertyCard;