import React, { useState, useEffect } from "react";
import { Property } from "../types";
import {
  X,
  MapPin,
  Square,
  Bed,
  Calendar,
  Building2,
  Edit,
  Trash2,
  Share2,
  Play,
  Bath,
  Car,
  Wifi,
  Shield,
  Zap,
  Trees
} from "lucide-react";
import { useAzureBlobMedia } from '../../../../hooks/useAzureBlobMedia';
import { useLegalHealth } from '../../../../hooks/useLegalHealth';
import { LegalHealthStatus } from '../../../../components/legal/LegalHealthStatus';

interface PropertyDetailsModalProps {
  property: Property | null;
  isOpen: boolean;
  onClose: () => void;
  onEdit?: (id: string) => void;
  onDelete?: (id: string) => void;
}

const PropertyDetailsModal: React.FC<PropertyDetailsModalProps> = ({
  property,
  isOpen,
  onClose,
  onEdit,
  onDelete,
}) => {
  const [videos, setVideos] = useState<any[]>([]);
  const [activeTab, setActiveTab] = useState("overview");
  const [currentMediaIndex, setCurrentMediaIndex] = useState(0);

  // Azure Blob media integration - only when property exists
  const {
    images: azureImages,
    videos: azureVideos,
    loading: mediaLoading,
    hasMedia
  } = useAzureBlobMedia(property?.id ? property.id.toString() : '');

  // Legal health integration
  const { legalHealth, loading: legalLoading } = useLegalHealth(property?.id || '');

  // Process images
  const allImages = (() => {
    const images = [];
    
    // Prioritize Azure Blob images if available
    if (azureImages && azureImages.length > 0) {
      azureImages.forEach((img) => {
        images.push(img.url);
      });
      return [...new Set(images)];
    }
    
    return [];
  })();

  // Initialize media display when Azure images are loaded
  useEffect(() => {
    if (allImages && allImages.length > 0) {
      setCurrentMediaIndex(0);
    }
  }, [allImages.length]);

  // Reset media state when modal closes
  useEffect(() => {
    if (!isOpen) {
      setCurrentMediaIndex(0);
      setActiveTab("overview");
    }
  }, [isOpen]);

  if (!isOpen || !property) return null;

  const handleEdit = () => {
    if (onEdit) {
      onEdit(property.id);
    }
    onClose();
  };

  const handleDelete = () => {
    if (onDelete && window.confirm('Are you sure you want to delete this property? This action cannot be undone.')) {
      onDelete(property.id);
    }
  };

  const formatPrice = (price: number, intent: string) => {
    if (!price) return "Price not specified";
    if (intent === 'rent') {
      return `₹${price?.toLocaleString()}/month`;
    }
    return `₹${(price / 100000)?.toFixed(1)} Lakhs`;
  };

  return (
    <div className="fixed inset-0 bg-black/50 z-50 overflow-y-auto">
      <div className="min-h-screen flex items-start justify-center p-4 pt-8">
        <div className="bg-white rounded-2xl max-w-7xl w-full shadow-2xl overflow-hidden">
          {/* Header */}
          <div className="flex items-center justify-between p-6 border-b bg-gray-50">
            <div>
              <h1 className="text-2xl font-bold text-gray-900">{property.title}</h1>
              <div className="flex items-center text-gray-600 mt-1">
                <MapPin className="w-4 h-4 mr-1" />
                <span className="text-sm">{property.address}, {property.city}</span>
              </div>
            </div>
            <div className="flex items-center space-x-3">
              {onEdit && (
                <button
                  onClick={handleEdit}
                  className="flex items-center space-x-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                >
                  <Edit className="w-4 h-4" />
                  <span>Edit</span>
                </button>
              )}
              {onDelete && (
                <button
                  onClick={handleDelete}
                  className="flex items-center space-x-2 px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors"
                >
                  <Trash2 className="w-4 h-4" />
                  <span>Delete</span>
                </button>
              )}
              <button
                onClick={onClose}
                className="p-2 text-gray-400 hover:text-gray-600 transition-colors"
              >
                <X size={24} />
              </button>
            </div>
          </div>

          {/* Main Content */}
          <div className="p-8">
            {/* Media Gallery */}
            <div className="relative bg-gray-100 rounded-xl overflow-hidden shadow-lg mb-8">
              <div className="relative h-80 w-full">
                <img
                  src={allImages.length > 0 ? allImages[currentMediaIndex] : "https://images.unsplash.com/photo-1586023492125-27b2c045efd7?ixlib=rb-4.0.3&auto=format&fit=crop&w=800&h=400"}
                  alt={property.title}
                  className="w-full h-full object-cover"
                  onError={(e) => {
                    e.currentTarget.src = "https://images.unsplash.com/photo-1586023492125-27b2c045efd7?ixlib=rb-4.0.3&auto=format&fit=crop&w=800&h=400";
                  }}
                />
                
                {/* Navigation Arrows */}
                {allImages.length > 1 && (
                  <>
                    <button
                      onClick={() => {
                        const newIndex = currentMediaIndex > 0 ? currentMediaIndex - 1 : allImages.length - 1;
                        setCurrentMediaIndex(newIndex);
                      }}
                      className="absolute left-3 top-1/2 transform -translate-y-1/2 bg-black/60 hover:bg-black/80 text-white p-3 rounded-full transition-all duration-300 z-10"
                    >
                      <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
                      </svg>
                    </button>
                    <button
                      onClick={() => {
                        const newIndex = currentMediaIndex < allImages.length - 1 ? currentMediaIndex + 1 : 0;
                        setCurrentMediaIndex(newIndex);
                      }}
                      className="absolute right-3 top-1/2 transform -translate-y-1/2 bg-black/60 hover:bg-black/80 text-white p-3 rounded-full transition-all duration-300 z-10"
                    >
                      <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
                      </svg>
                    </button>
                  </>
                )}

                {/* Media Counter */}
                {allImages.length > 0 && (
                  <div className="absolute bottom-4 left-4 bg-black/70 text-white px-3 py-1 rounded-full text-sm">
                    {currentMediaIndex + 1} / {allImages.length}
                  </div>
                )}
              </div>
            </div>

            {/* Property Details */}
            <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
              <div className="lg:col-span-2">
                <div className="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-xl p-6 mb-6">
                  <h3 className="text-3xl font-bold text-gray-900 mb-2">
                    {formatPrice(property.price, property.listingIntent)}
                  </h3>
                  <div className="grid grid-cols-4 gap-4 mt-4">
                    <div className="text-center p-3 bg-white rounded-lg">
                      <Bed className="w-6 h-6 text-blue-600 mx-auto mb-2" />
                      <div className="text-lg font-bold">{property.bhk?.replace(' BHK', '') || 'N/A'}</div>
                      <div className="text-xs text-gray-600">Bedrooms</div>
                    </div>
                    <div className="text-center p-3 bg-white rounded-lg">
                      <Bath className="w-6 h-6 text-blue-600 mx-auto mb-2" />
                      <div className="text-lg font-bold">{property.bathrooms || '2'}</div>
                      <div className="text-xs text-gray-600">Bathrooms</div>
                    </div>
                    <div className="text-center p-3 bg-white rounded-lg">
                      <Square className="w-6 h-6 text-blue-600 mx-auto mb-2" />
                      <div className="text-lg font-bold">{property.area || 'N/A'}</div>
                      <div className="text-xs text-gray-600">Sq Ft</div>
                    </div>
                    <div className="text-center p-3 bg-white rounded-lg">
                      <Building2 className="w-6 h-6 text-blue-600 mx-auto mb-2" />
                      <div className="text-lg font-bold capitalize">{property.type || 'N/A'}</div>
                      <div className="text-xs text-gray-600">Type</div>
                    </div>
                  </div>
                </div>

                <div className="space-y-6">
                  <div>
                    <h3 className="text-xl font-semibold text-gray-900 mb-3">Property Description</h3>
                    <p className="text-gray-700 leading-relaxed">
                      {property.description || 'No description available for this property.'}
                    </p>
                  </div>

                  {/* Legal Health Section */}
                  {legalHealth && (
                    <LegalHealthStatus 
                      legalHealth={legalHealth}
                      showDetails={true}
                      userRole="seller"
                      onUploadClick={() => {
                        // Navigate to legal documents page
                        window.location.href = `/seller/dashboard/legal-documents`;
                      }}
                    />
                  )}
                </div>
              </div>

              <div className="space-y-6">
                <div className="bg-gradient-to-br from-blue-600 to-blue-700 rounded-xl p-6 text-white">
                  <h3 className="text-lg font-semibold mb-4">Property Actions</h3>
                  <div className="space-y-3">
                    {onEdit && (
                      <button
                        onClick={handleEdit}
                        className="w-full bg-white/20 hover:bg-white/30 py-3 rounded-lg font-medium transition-colors"
                      >
                        Edit Property
                      </button>
                    )}
                    {onDelete && (
                      <button
                        onClick={handleDelete}
                        className="w-full bg-red-500 hover:bg-red-600 py-3 rounded-lg font-medium transition-colors"
                      >
                        Delete Property
                      </button>
                    )}
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default PropertyDetailsModal;